/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;

/**
 * <p>Title: ElementCylinder</p>
 * <p>Description: Painter's algorithm implementation of a Cylinder</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementCylinder extends AbstractTile implements org.opensourcephysics.display3d.core.ElementCylinder {
  // Configuration variables
  private boolean closedBottom = true, closedTop = true;
  private boolean closedLeft   = true, closedRight = true;
  private int minAngle = 0, maxAngle = 360;

  // Implementation variables
  private boolean changeNTiles = true;
  private int nr = -1, nu = -1, nz = -1; // Make sure arrays are allocated
  private double[][][] standardCylinder = null;

  { // Initialization block
    getStyle().setResolution(new Resolution (3,12,5));
  }

// -------------------------------------
// Configuration
// -------------------------------------

  public void setClosedBottom (boolean close) {
    this.closedBottom = close;
    setElementChanged(true);
    changeNTiles = true;
  }
  public boolean isClosedBottom () { return this.closedBottom; }

  public void setClosedTop (boolean close) {
    this.closedTop = close;
    setElementChanged(true);
    changeNTiles = true;
  }
  public boolean isClosedTop () { return this.closedTop; }

  public void setClosedLeft (boolean close) {
    this.closedLeft = close;
    setElementChanged(true);
    changeNTiles = true;
  }
  public boolean isClosedLeft () { return this.closedLeft; }

  public void setClosedRight (boolean close) {
    this.closedRight = close;
    setElementChanged(true);
    changeNTiles = true;
  }
  public boolean isClosedRight () { return this.closedRight; }

  public void setMinimumAngle(int angle)  {
    this.minAngle = angle;
    setElementChanged(true);
    changeNTiles = true;
  }
  public int getMinimumAngle() { return this.minAngle; }

  public void setMaximumAngle(int angle)  {
    this.maxAngle = angle;
    setElementChanged(true);
    changeNTiles = true;
  }
  public int getMaximumAngle() { return this.maxAngle; }

// -------------------------------------
//  Private or protected methods
// -------------------------------------

  protected synchronized void computeCorners () {
    int theNr = 1, theNu = 1, theNz = 1;
    double angle1 = minAngle, angle2 = maxAngle;
    if (Math.abs(angle2-angle1)>360) angle2 = angle1+360;
    org.opensourcephysics.display3d.core.Resolution res = getRealStyle().getResolution();
    if (res!=null) {
      switch (res.getType()) {
        case Resolution.DIVISIONS :
          theNr = Math.max(res.getN1(),1);
          theNu = Math.max(res.getN2(),1);
          theNz = Math.max(res.getN3(),1);
          break;
        case Resolution.MAX_LENGTH :
          double dx = Math.abs(getSizeX())/2, dy = Math.abs(getSizeY())/2;
          theNr = Math.max((int) Math.round(0.49 + Math.max(dx,dy)/res.getMaxLength()), 1);
          theNu = Math.max((int) Math.round(0.49 + Math.abs(angle2-angle1)*TO_RADIANS*(dx+dy)/res.getMaxLength()), 1);
          theNz = Math.max((int) Math.round(0.49 + Math.abs(getSizeZ())/res.getMaxLength()), 1);
        break;
      }
    }
    if (nr!=theNr || nu!=theNu || nz!=theNz || changeNTiles) { // Reallocate arrays
      nr = theNr; nu = theNu; nz = theNz;
      changeNTiles = false;
      standardCylinder = createStandardCylinder (nr, nu, nz, angle1, angle2, closedTop, closedBottom, closedLeft, closedRight);
      setCorners(new double [standardCylinder.length][4][3]);
    }
    for (int i=0; i<numberOfTiles; i++)
      for (int j=0, sides=corners[i].length; j<sides; j++) {
        System.arraycopy(standardCylinder[i][j], 0, corners[i][j], 0, 3);
        toSpace(corners[i][j]);
      }
    setElementChanged(false);
  }

  static final protected double TO_RADIANS=Math.PI/180.0;
  static final protected double[] vectorx = {1.0,0.0,0.0}, // Standard vertical Cylinder
                                  vectory = {0.0,1.0,0.0},
                                  vectorz = {0.0,0.0,1.0};

  static private double[][][] createStandardCylinder (int nr, int nu, int nz, double angle1, double angle2,
                                                      boolean top, boolean bottom, boolean left, boolean right) {
    int totalN = nu*nz;
    if (bottom) totalN += nr*nu;
    if (top)    totalN += nr*nu;
    if (Math.abs(angle2-angle1)<360) {
      if (left)   totalN += nr*nz;
      if (right)  totalN += nr*nz;
    }
    double[][][] data = new double [totalN][4][3];
    // Compute sines and cosines
    double[] cosu = new double[nu+1], sinu = new double[nu+1];
    for (int u=0; u<=nu; u++) { // compute sines and cosines
      double angle = ((nu-u)*angle1 + u*angle2)*TO_RADIANS/nu;
      cosu[u] = Math.cos(angle)/2; // The /2 is because the element is centered
      sinu[u] = Math.sin(angle)/2;
    }
    // Now compute the tiles
    int tile = 0;
    double[] center =  new double[]{-vectorz[0]/2,-vectorz[1]/2,-vectorz[2]/2};
    { // Tiles along the z axis
      double aux = 1.0/nz;
      for (int j=0; j<nz; j++) {
        for (int u=0; u<nu; u++, tile++) { // This ordering is important for the computations below (see ref)
          for (int k=0; k<3; k++) {
            data[tile][0][k] = center[k] + cosu[u  ]*vectorx[k] + sinu[u  ]*vectory[k] + j    *aux*vectorz[k];
            data[tile][1][k] = center[k] + cosu[u+1]*vectorx[k] + sinu[u+1]*vectory[k] + j    *aux*vectorz[k];
            data[tile][2][k] = center[k] + cosu[u+1]*vectorx[k] + sinu[u+1]*vectory[k] + (j+1)*aux*vectorz[k];
            data[tile][3][k] = center[k] + cosu[u  ]*vectorx[k] + sinu[u  ]*vectory[k] + (j+1)*aux*vectorz[k];
          }
        }
      }
    }
    if (bottom) { // Tiles at bottom
      int ref=0; // not used
      for (int u=0; u<nu; u++) {
        for (int i=0; i<nr; i++, tile++) {
          for (int k=0; k<3; k++) {
            data[tile][0][k] = ((nr-i)  *center[k] +  i   *data[u][0][k])/nr; // should be ref+u
            data[tile][1][k] = ((nr-i-1)*center[k] + (i+1)*data[u][0][k])/nr; // should be ref+u
            data[tile][2][k] = ((nr-i-1)*center[k] + (i+1)*data[u][1][k])/nr; // should be ref+u
            data[tile][3][k] = ((nr-i)  *center[k] +  i   *data[u][1][k])/nr; // should be ref+u
          }
        }
      }
    }
    if (top) { // Tiles at top
      int ref = nu*(nz-1);
      center[0] = vectorz[0]; center[1] = vectorz[1]; center[2] = vectorz[2]-0.5;
      for (int u=0; u<nu; u++) {
        for (int i=0; i<nr; i++, tile++) {
          for (int k=0; k<3; k++) {
            data[tile][0][k] = ((nr-i)  *center[k] +  i   *data[ref+u][3][k])/nr;
            data[tile][1][k] = ((nr-i-1)*center[k] + (i+1)*data[ref+u][3][k])/nr;
            data[tile][2][k] = ((nr-i-1)*center[k] + (i+1)*data[ref+u][2][k])/nr;
            data[tile][3][k] = ((nr-i)  *center[k] +  i   *data[ref+u][2][k])/nr;
          }
        }
      }
    }
    if (Math.abs(angle2-angle1)<360){ // No need to close left or right if the Cylinder is 'round' enough
      center[0] = -vectorz[0]/2; center[1] = -vectorz[1]/2; center[2] = -vectorz[2]/2;
      if (right) { // Tiles at right
        double aux = 1.0/nz;
        for (int j=0; j<nz; j++) {
          for (int i=0; i<nr; i++, tile++) {
            for (int k=0; k<3; k++) {
              data[tile][0][k] = ((nr-i)  *center[k] +  i   *data[0][0][k])/nr + j    *aux*vectorz[k];
              data[tile][1][k] = ((nr-i-1)*center[k] + (i+1)*data[0][0][k])/nr + j    *aux*vectorz[k];
              data[tile][2][k] = ((nr-i-1)*center[k] + (i+1)*data[0][0][k])/nr + (j+1)*aux*vectorz[k];
              data[tile][3][k] = ((nr-i)  *center[k] +  i   *data[0][0][k])/nr + (j+1)*aux*vectorz[k];
            }
          }
        }
      }
      if (left) { // Tiles at left
        double aux = 1.0/nz;
        int ref = nu-1;
        for (int j=0; j<nz; j++) {
          for (int i=0; i<nr; i++, tile++) {
            for (int k=0; k<3; k++) {
              data[tile][0][k] = ((nr-i)  *center[k] +  i   *data[ref][1][k])/nr + j    *aux*vectorz[k];
              data[tile][1][k] = ((nr-i-1)*center[k] + (i+1)*data[ref][1][k])/nr + j    *aux*vectorz[k];
              data[tile][2][k] = ((nr-i-1)*center[k] + (i+1)*data[ref][1][k])/nr + (j+1)*aux*vectorz[k];
              data[tile][3][k] = ((nr-i)  *center[k] +  i   *data[ref][1][k])/nr + (j+1)*aux*vectorz[k];
            }
          }
        }
      }
    }
    return data;
  }


// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementCylinderLoader(); }

  static protected class ElementCylinderLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementCylinder(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementCylinder element = (ElementCylinder) obj;
      control.setValue("closed top",    element.closedTop);
      control.setValue("closed bottom", element.closedBottom);
      control.setValue("closed left",   element.closedLeft);
      control.setValue("closed right",  element.closedRight);
      control.setValue("minimum angle",  element.minAngle);
      control.setValue("maximum angle",  element.maxAngle);
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementCylinder element = (ElementCylinder) obj;
      element.closedTop    = control.getBoolean("closed top");
      element.closedBottom = control.getBoolean("closed bottom");
      element.closedLeft   = control.getBoolean("closed left");
      element.closedRight  = control.getBoolean("closed right");
      element.minAngle    = control.getInt("minimum angle");
      element.maxAngle    = control.getInt("maximum angle");
      element.changeNTiles = true;
      return obj;
    }

  }

}

