/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import org.opensourcephysics.controls.XML;
import org.opensourcephysics.controls.XMLControl;

/**
 * <p>Title: ElementArrow</p>
 * <p>Description: An Arrow using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementArrow extends ElementSegment implements org.opensourcephysics.display3d.core.ElementArrow {

// -------------------------------------
// Super methods overwritten
// -------------------------------------

  void draw (Graphics2D _g2, int _index) {
    // Allow the panel to adjust color according to depth
    Color theColor = getPanel().projectColor(getRealStyle().getLineColor(),objects[_index].getDistance());
    if (_index<(div-1)) {
      _g2.setStroke(getRealStyle().getLineStroke());
      _g2.setColor (theColor);
      _g2.drawLine ((int) aCoord[_index], (int) bCoord[_index], (int) aCoord[_index+1], (int) bCoord[_index+1]);
    }
    else { // Draw the head
      Color theFillColor = getPanel().projectColor(getRealStyle().getFillColor(),objects[_index].getDistance());
      drawHead (_g2,(int) aCoord[_index],(int) bCoord[_index],theColor,theFillColor);
    }
  }

  synchronized void drawQuickly (Graphics2D _g2) {
    if (!isVisible()) return;
    if (hasChanged()) { computeDivisions(); projectPoints(); }
    else if (needsToProject()) projectPoints();
    drawHead (_g2,(int) aCoord[0], (int) bCoord[0],getRealStyle().getLineColor(),getRealStyle().getFillColor());
  }

// -------------------------------------
// The head
// -------------------------------------

  static final private double ARROW_CST=0.35;
  static final private double ARROW_MAX=25.0;

  private int headPoints=0;
  private int headA[] = new int [10], headB[] = new int [10]; // Used to display the head

  void projectPoints () {
    super.projectPoints();
    // Now compute the head
    double a = aCoord[div] - aCoord[0];
    double b = bCoord[div] - bCoord[0];
    double h = Math.sqrt (a*a+b*b);
//FKH 20020331
    if (h==0.0) { headPoints = 0; return; }
    a = ARROW_CST*a / h; b = ARROW_CST*b / h;
    if(h>ARROW_MAX){ a*=ARROW_MAX/h; b*=ARROW_MAX/h;}
    int p0 = (int) (aCoord[div] - a*h);
    int q0 = (int) (bCoord[div] - b*h);
    a *= h/2.0; b *= h/2.0;
    headPoints = 6;
    headA[0] = p0;                 headB[0] = q0;
    headA[1] = p0-(int)b;          headB[1] = q0+(int)a;
    headA[2] = aCoord[div];        headB[2] = bCoord[div];
    headA[3] = p0+(int)b;          headB[3] = q0-(int)a;
    headA[4] = p0;                 headB[4] = q0;
  }

  private void drawHead (Graphics2D _g2, int a1, int b1, Color _color, Color _fill) {
    _g2.setStroke (getRealStyle().getLineStroke());
    if (headPoints==0) {
      _g2.setColor (_color);
      _g2.drawLine (a1,b1,aCoord[div],bCoord[div]);
      return;
   }
   int n = headPoints-1;
   headA[n] = a1;
   headB[n] = b1;
   if (_fill!=null) {
     _g2.setPaint(_fill);
     _g2.fillPolygon(headA, headB, n);
   }
   _g2.setColor (_color);
   _g2.drawPolyline (headA,headB,headPoints);
 }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementArrowLoader(); }

  static private class ElementArrowLoader extends ElementLoader {
    public Object createObject(XMLControl control) { return new ElementArrow(); }
  }

}
