/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;
import org.opensourcephysics.numerics.Transformation;

public abstract class Element implements org.opensourcephysics.display3d.core.Element {
  // Configuration variables
  private boolean visible = true;  // is the object visible?
  private double x=0.0, y=0.0, z=0.0; // position of the element
  private double sizeX=1.0, sizeY=1.0, sizeZ=1.0; // the size of the element in each dimension
  private String name = "";
  private Transformation transformation = null;
  private Style style=new Style(this);
  private Group group=null;

  // Implementation variables
  private boolean elementChanged=true, needsToProject=true;
  private DrawingPanel3D panel;

  /**
   * Returns the DrawingPanel3D in which it (or its final ancestor group)
   * is displayed.
   * @return DrawingPanel3D
   */
  final DrawingPanel3D getPanel() {
    Element el = this;
    while (el.group!=null) el = el.group;
    return el.panel;
  }

  /**
   * To be used internally by DrawingPanel3D only! Sets the panel for this element.
   * @param _panel DrawingPanel3D
   */
  void setPanel(DrawingPanel3D _panel) { this.panel = _panel; elementChanged = true; }

  /**
   * Returns the group to which the element belongs
   * @return Group Returns null if it doesn't belong to a group
   */
  final Group getGroup() { return group; }

  /**
   * To be used internally by Group only! Sets the group of this element.
   * @param _group Group
   */
  void setGroup(Group _group) { this.group = _group; elementChanged = true; }

// ----------------------------------------
// Name of the element
// ----------------------------------------

  public void setName (String aName) { this.name = aName; }

  final public String getName() { return this.name; }

// ----------------------------------------
// Position of the element
// ----------------------------------------

  public void setX(double x) { this.x = x; elementChanged = true; }
  final public double getX() { return this.x; }

  public void setY(double y) { this.y = y; elementChanged = true; }
  final public double getY() { return this.y; }

  public void setZ(double z) { this.z = z; elementChanged = true; }
  final public double getZ() { return this.z; }

  public void setXYZ(double x, double y, double z) {
    this.x = x; this.y = y; this.z = z; elementChanged = true;
  }

  /**
   * Returns the extreme points of a box that contains the element.
   * @param min double[] A previously allocated double[3] array that will hold
   * the minimum point
   * @param max double[] A previously allocated double[3] array that will hold
   * the maximum point
   */
  void getExtrema (double[] min, double[] max) {
    min[0] = -0.5; max[0] = 0.5;
    min[1] = -0.5; max[1] = 0.5;
    min[2] = -0.5; max[2] = 0.5;
    toSpace(min);
    toSpace(max);
  }

// ----------------------------------------
// Size of the element
// ----------------------------------------

  public void setSizeX(double sizeX) { this.sizeX = sizeX; elementChanged = true; }
  final public double getSizeX() { return this.sizeX; }

  public void setSizeY(double sizeY) { this.sizeY = sizeY; elementChanged = true; }
  final public double getSizeY() { return this.sizeY; }

  public void setSizeZ(double sizeZ) { this.sizeZ = sizeZ; elementChanged = true; }
  final public double getSizeZ() { return this.sizeZ; }

  public void setSizeXYZ(double sizeX, double sizeY, double sizeZ) {
    this.sizeX = sizeX; this.sizeY = sizeY; this.sizeZ = sizeZ; elementChanged = true;
  }

  /**
   * Returns the diagonal size of the element, i.e., Math.sqrt(sizeX*sizeX+sizeY*sizeY+sizeZ*sizeZ)
   * @return double
   */
  final double getDiagonalSize () { return Math.sqrt(sizeX*sizeX+sizeY*sizeY+sizeZ*sizeZ); }

  /**
   * Returns whether the element has changed significantly.
   * This can be used by implementing classes to help improve performance.
   * @return boolean
   */
  final boolean hasChanged () {
    Element el = this;
    while (el!=null) {
      if (el.elementChanged) return true;
      el = el.group;
    }
    return false;
  }

  /**
   * Tells the element whether it has a change that demands some
   * kind of computation or, the other way round, someone took
   * care of all possible changes.
   * Typically used by subclasses when they change something or
   * make all needed computations.
   * @param change Whether the element has changed
   */
  final void setElementChanged (boolean change) { elementChanged = change; }

// -------------------------------------
// Visibility and style
// -------------------------------------

  public void setVisible (boolean _visible) { this.visible = _visible; }
  final public boolean isVisible () { return this.visible; }

  final public org.opensourcephysics.display3d.core.Style getStyle ()   { return this.style; }

  /**
   * Gets the real Style. This is more convenient and improves performance (a liiittle bit)
   * @return Style
   */
  final Style getRealStyle ()   { return this.style; }

  /**
   * Used by Style to notify possible changes.
   * @param styleThatChanged int
   */
  abstract void styleChanged (int styleThatChanged);

// ----------------------------------------
// Transformation of the element
// ----------------------------------------

  public void setTransformation (org.opensourcephysics.numerics.Transformation transformation) {
    if (transformation==null) this.transformation = null;
    else this.transformation = (Transformation) transformation.clone();
    elementChanged = true;
  }

  public void toSpaceFrame (double[] vector) {
    Element el = this;
    do {
      if (el.transformation!=null) el.transformation.direct(vector);
      vector[0] += el.x; vector[1] += el.y; vector[2] += el.z;
      el = el.group;
    } while (el!=null);
  }

  public void toBodyFrame (double[] vector) throws UnsupportedOperationException {
    java.util.ArrayList elList = new java.util.ArrayList();
    Element el = this;
    do { elList.add(el); el = el.group; } while (el!=null);
    for (int i=elList.size()-1; i>=0; i--) { // Done in the reverse order
      el = (Element) elList.get(i);
      vector[0] -= el.x; vector[1] -= el.y; vector[2] -= el.z;
      if (el.transformation!=null) el.transformation.inverse(vector);
    }
  }

  /**
   * Translates a point of the standard (0,0,0) to (1,1,1) element
   * to its real spatial coordinate. Thus, if the point has a coordinate of 1,
   * the result will be the size of the element.
   * It is package-private and is VERY useful to compute the real position
   * of a point of the element in space
   * @param vector the vector to be converted
   */
  final void toSpace (double[] vector) {
    Element el = this;
    do {
      vector[0] *= el.sizeX; vector[1] *= el.sizeY; vector[2] *= el.sizeZ;
      if (el.transformation!=null) el.transformation.direct(vector);
      vector[0] += el.x; vector[1] += el.y; vector[2] += el.z;
      el = el.group;
    } while (el!=null);
  }

  /**
   * Translates a point of the standard (0,0,0) to (sizex,sizey,sizez) element
   * to its real spatial coordinate. Thus, if the point has a coordinate of 1,
   * the result will be the size of the element.
   * It is package-private and is needed for those element which compute
   * their points using their real size (such as ElementSpring, to name one)
   * @param vector the vector to be converted
   */
  final void toSpaceSized (double[] vector) {
    if (transformation!=null) transformation.direct(vector);
    vector[0] += x; vector[1] += y; vector[2] += z;
    Element el = group;
    while (el!=null) {
      vector[0] *= el.sizeX; vector[1] *= el.sizeY; vector[2] *= el.sizeZ;
      if (el.transformation!=null) el.transformation.direct(vector);
      vector[0] += el.x; vector[1] += el.y; vector[2] += el.z;
      el = el.group;
    }
  }

// ----------------------------------------------------
// Needed by the drawing mechanism
// ----------------------------------------------------
  /**
   * Returns an array of Objects3D to sort according to its distance and draw.
   */
  abstract Object3D[] getObjects3D ();

  /**
   * Draws a given Object3D (indicated by its index).
   */
  abstract void draw (java.awt.Graphics2D g, int index);

  /**
   * Sketches the drawable
   */
  abstract void drawQuickly (java.awt.Graphics2D g);

  /**
   * Tells the element whether it should reproject its points because the panel
   * has changed its projection parameters. Or, the other way round,
   * if someone (typically methods in subclasses) took care of this already.
   */
  void setNeedToProject(boolean _need) { needsToProject = _need; }

  /**
   * Whether the element needs to project
   * @return boolean
   * @see #setNeedToProject(boolean)
   */
  final boolean needsToProject () { return needsToProject; }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * A class to save and load Element data.
   */
  static protected abstract class ElementLoader implements XML.ObjectLoader {
    public void saveObject(XMLControl control, Object obj) {
      Element element = (Element) obj;
      if (element.name.length()>0) control.setValue("name", element.name);
      control.setValue("x", element.x);
      control.setValue("y", element.y);
      control.setValue("z", element.z);
      control.setValue("x size", element.sizeX);
      control.setValue("y size", element.sizeY);
      control.setValue("z size", element.sizeZ);
      control.setValue("visible", element.visible);
      control.setValue("style", element.style);
      control.setValue("transformation", element.transformation);
   }

    public abstract Object createObject(XMLControl control);

    public Object loadObject(XMLControl control, Object obj) {
      Element element = (Element) obj;
      String name = control.getString("name");
      if (name!=null) element.setName(name);
      element.x = control.getDouble("x");
      element.y = control.getDouble("y");
      element.z = control.getDouble("z");
      element.sizeX = control.getDouble("x size");
      element.sizeY = control.getDouble("y size");
      element.sizeZ = control.getDouble("z size");
      element.visible = control.getBoolean("visible");
      element.style = (Style) control.getObject("style");
      element.style.setElement(element);
//      element.styleChanged(Style.STYLE_RESOLUTION); --> elementChanged = true;
      element.transformation = (Transformation) control.getObject("transformation"); // no need to clone
      element.elementChanged = true;
      return obj;
    }
  } // End of static class ElementLoader

}













