/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.display.MouseController;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import org.opensourcephysics.controls.*;

public class DrawingPanel3D extends javax.swing.JPanel
                            implements org.opensourcephysics.display3d.core.DrawingPanel3D, Printable {

  static public int UPDATE_AUTOMATIC = org.opensourcephysics.display3d.core.DrawingPanel3D.UPDATE_AUTOMATIC;
  static public int UPDATE_MANUAL = org.opensourcephysics.display3d.core.DrawingPanel3D.UPDATE_MANUAL;

  static private final int AXIS_DIVISIONS = 10;

  // Configuration variables
  private int updateStrategy = UPDATE_MANUAL;
  private double xmin=-1.0, xmax= 1.0, ymin=-1.0, ymax= 1.0, zmin=-1.0, zmax= 1.0;
  private VisualizationHints visHints=null;

  // Implementation variables
  private boolean quickRedrawOn = false, squareAspect = true, needsToRecompute=true;
  private double cosAlpha, sinAlpha, cosBeta, sinBeta;
  private double ratioToPlane=2.5, ratioToCenter=2.0;
  private double centerX, centerY, centerZ;
  private double aconstant, bconstant, viewToPlane, viewToCenter;
  private int acenter, bcenter;
  private Color bgColor = new Color(239, 239, 255);
  private ArrayList list3D = new ArrayList();
  private ArrayList decorationList = new ArrayList();
  private ArrayList elementList = new ArrayList();
  private Object3D.Comparator3D comparator = new Object3D.Comparator3D(); // see class Comparator3D below

  // Variables for decoration
  private ElementArrow xAxis, yAxis, zAxis;
  private ElementText  xText, yText, zText;
  private ElementSegment[] boxSides = new ElementSegment[12];

  public DrawingPanel3D() {
    setBackground(bgColor);
    setPreferredSize(new Dimension(300, 300));
    visHints = new VisualizationHints(this);
    cosAlpha=Math.cos(visHints.getAlpha()); sinAlpha=Math.sin(visHints.getAlpha());
    cosBeta =Math.cos(visHints.getBeta());  sinBeta =Math.sin(visHints.getBeta());
    addComponentListener (
      new java.awt.event.ComponentAdapter () {
       public void componentResized (java.awt.event.ComponentEvent e) { computeConstants (); }
     }
    );
    IADMouseController mouseController = new IADMouseController();
    addMouseListener(mouseController);
    addMouseMotionListener(mouseController);
    /* Decoration of the scene */
    // Create the bounding box
    Resolution axesRes = new Resolution(AXIS_DIVISIONS,1,1);
    for (int i=0, n=boxSides.length; i<n; i++) {
      boxSides[i] = new ElementSegment();
      boxSides[i].getRealStyle().setResolution(axesRes);
      boxSides[i].setPanel(this); // Because I don't add it to the panel in the standard way
      decorationList.add(boxSides[i]);
    }
    boxSides[0].getStyle().setLineColor(new Color (128,0,0));
    boxSides[3].getStyle().setLineColor(new Color (0,128,0));
    boxSides[8].getStyle().setLineColor(new Color (0,0,255));
    // Create the labelled axes
    xAxis = new ElementArrow();
    xAxis.getRealStyle().setResolution(axesRes);
    xAxis.getStyle().setFillColor(new Color(128,0,0));
    xAxis.setPanel(this);
    decorationList.add(xAxis);
    xText = new ElementText ();
    xText.setText("X");
    xText.setJustification(ElementText.JUSTIFICATION_CENTER);
    xText.getRealStyle().setLineColor(Color.BLACK);
    xText.setFont(new Font("Dialog",Font.PLAIN,12));
    xText.setPanel(this);
    decorationList.add(xText);
    yAxis = new ElementArrow();
    yAxis.getRealStyle().setResolution(axesRes);
    yAxis.getStyle().setFillColor(new Color(0,128,0));
    yAxis.setPanel(this);
    decorationList.add(yAxis);
    yText = new ElementText ();
    yText.setText("Y");
    yText.setJustification(ElementText.JUSTIFICATION_CENTER);
    yText.getRealStyle().setLineColor(Color.BLACK);
    yText.setFont(new Font("Dialog",Font.PLAIN,12));
    yText.setPanel(this);
    decorationList.add(yText);
    zAxis = new ElementArrow();
    zAxis.getRealStyle().setResolution(axesRes);
    zAxis.getStyle().setFillColor(new Color(0,0,255));
    zAxis.setPanel(this);
    decorationList.add(zAxis);
    zText = new ElementText ();
    zText.setText("Z");
    zText.setJustification(ElementText.JUSTIFICATION_CENTER);
    zText.getRealStyle().setLineColor(Color.BLACK);
    zText.setFont(new Font("Dialog",Font.PLAIN,12));
    zText.setPanel(this);
    decorationList.add(zText);
    /* End of decoration */
    // Set default for displayMode
    if (visHints.getDisplayMode()<VisualizationHints.DISPLAY_3D) { // i.e. 2D
      visHints.setDecorationType(VisualizationHints.DECORATION_NONE);
      visHints.setUseColorDepth(false);
    }
    else {
      visHints.setDecorationType(VisualizationHints.DECORATION_CUBE);
      visHints.setUseColorDepth(true);
    }
    computeConstants();
  }

// ---------------------------------
// Implementation of core.DrawinPanel3D
// ---------------------------------

  public javax.swing.JPanel getJPanel() { return this; }

  public void setPreferredMinMax (double minX, double maxX, double minY, double maxY, double minZ, double maxZ) {
    this.xmin = minX; this.xmax = maxX;
    this.ymin = minY; this.ymax = maxY;
    this.zmin = minZ; this.zmax = maxZ;
    needsToRecompute = true;
    if (updateStrategy==UPDATE_AUTOMATIC) updatePanel();
  }
  public double getPreferredMinX () { return this.xmin; }
  public double getPreferredMaxX () { return this.xmax; }
  public double getPreferredMinY () { return this.ymin; }
  public double getPreferredMaxY () { return this.ymax; }
  public double getPreferredMinZ () { return this.zmin; }
  public double getPreferredMaxZ () { return this.zmax; }

  public void zoomToFit() {
    double minX = Double.POSITIVE_INFINITY, maxX = Double.NEGATIVE_INFINITY;
    double minY = Double.POSITIVE_INFINITY, maxY = Double.NEGATIVE_INFINITY;
    double minZ = Double.POSITIVE_INFINITY, maxZ = Double.NEGATIVE_INFINITY;
    double [] firstPoint = new double[3], secondPoint = new double[3];
    Iterator it = getElements().iterator();
    while (it.hasNext()) {
      ((Element) it.next()).getExtrema (firstPoint,secondPoint);
      minX = Math.min (Math.min (minX,firstPoint[0]),secondPoint[0]);
      maxX = Math.max (Math.max (maxX,firstPoint[0]),secondPoint[0]);
      minY = Math.min (Math.min (minY,firstPoint[1]),secondPoint[1]);
      maxY = Math.max (Math.max (maxY,firstPoint[1]),secondPoint[1]);
      minZ = Math.min (Math.min (minZ,firstPoint[2]),secondPoint[2]);
      maxZ = Math.max (Math.max (maxZ,firstPoint[2]),secondPoint[2]);
    }
    double max = Math.max(Math.max(maxX-minX, maxY-minY),maxZ-minZ);
    if (max==0.0) max = 2;
    if (minX>=maxX) { minX = maxX-max/2; maxX = minX + max; }
    if (minY>=maxY) { minY = maxY-max/2; maxY = minY + max; }
    if (minZ>=maxZ) { minZ = maxZ-max/2; maxZ = minZ + max; }
    setPreferredMinMax (minX,maxX,minY,maxY,minZ,maxZ);
  }

  public void setSquareAspect (boolean square) {
    squareAspect = square;
    computeConstants();
  }

  public boolean isSquareAspect () { return squareAspect; }

  public org.opensourcephysics.display3d.core.VisualizationHints getVisualizationHints () { return visHints; }

  public void updatePanel() { repaint(); }

  public void setUpdateStrategy(int strategy) {
    updateStrategy = strategy;
    if (strategy==UPDATE_AUTOMATIC) updatePanel();
  }

  public int getUpdateStrategy () { return updateStrategy; }

  public void addElement(org.opensourcephysics.display3d.core.Element element) {
    if (! (element instanceof Element) )
      throw new UnsupportedOperationException("Can't add element to panel (incorrect implementation)");
    if (!elementList.contains(element)) elementList.add(element);
    ((Element) element).setPanel(this);
    if (updateStrategy==UPDATE_AUTOMATIC) updatePanel();
  }

  public void removeElement(org.opensourcephysics.display3d.core.Element element) {
    elementList.remove(element);
    if (updateStrategy==UPDATE_AUTOMATIC) updatePanel();
  }

  public void removeAllElements() {
    elementList.clear();
    if (updateStrategy==UPDATE_AUTOMATIC) updatePanel();
  }

  public synchronized ArrayList getElements() {
     return (ArrayList) elementList.clone();
  }

// ----------------------------------------------------
// All the painting stuff
// ----------------------------------------------------

  public void paintComponent(Graphics g) { paintEverything(g); }

  private void paintEverything(Graphics g) {
    if (needsToRecompute) computeConstants();
    ArrayList tempList = getElements();
    tempList.addAll(decorationList);
    g.setColor(getBackground());
    g.fillRect(0, 0, getWidth(), getHeight()); // fill the component with the background color
    paintDrawableList(g, tempList);
  }

  private void paintDrawableList(Graphics g, ArrayList tempList) {
    Graphics2D g2 = (Graphics2D) g;
    Iterator   it = tempList.iterator();
    if (quickRedrawOn || !visHints.isRemoveHiddenLines()) { // Do a quick sketch of the scene
      while (it.hasNext()) ((Element) it.next()).drawQuickly(g2);
      return;
    }
    // Collect objects, sort and draw them one by one. Takes time!!!
    list3D.clear();
    while (it.hasNext()) { // Collect all Objects3D
      Object3D[] objects = ((Element) it.next()).getObjects3D();
      if (objects==null) continue;
      for (int i=0, n=objects.length; i<n; i++) {
        // providing NaN as distance can be used by Drawables3D to hide a given Object3D
        if (!Double.isNaN(objects[i].getDistance())) list3D.add(objects[i]);
      }
    }
    if (list3D.size()<=0) return;
    Object[] objects = list3D.toArray();
    Arrays.sort(objects,comparator);
    for (int i=0, n=objects.length; i<n; i++) {
      Object3D obj = (Object3D) objects[i];
      obj.getElement().draw(g2,obj.getIndex());
    }
  }

// ----------------------------------------------------
// Printable interface
// ----------------------------------------------------

  public int print(Graphics g, PageFormat pageFormat, int pageIndex) throws PrinterException {
    if(pageIndex>=1) return Printable.NO_SUCH_PAGE;
    if(g==null) return Printable.NO_SUCH_PAGE;
    Graphics2D g2 = (Graphics2D) g;
    double scalex = pageFormat.getImageableWidth()/(double) getWidth();
    double scaley = pageFormat.getImageableHeight()/(double) getHeight();
    double scale = Math.min(scalex, scaley);
    g2.translate((int) pageFormat.getImageableX(), (int) pageFormat.getImageableY());
    g2.scale(scale, scale);
    paintEverything(g2);
    return Printable.PAGE_EXISTS;
  }

// ----------------------------------------------------
// Projection, package and private methods
// - ---------------------------------------------------

  /**
   * This will be called by VisualizationHints whenever hints change.
   * @see VisualizationHints
   */
  void hintChanged (int hintThatChanged) {
    switch (hintThatChanged) {
      case VisualizationHints.HINT_DISPLAY_MODE : computeConstants(); break;
      case VisualizationHints.HINT_DECORATION_TYPE :
        switch (visHints.getDecorationType()) {
          case VisualizationHints.DECORATION_NONE :
            for (int i=0,n=boxSides.length; i<n; i++) boxSides[i].setVisible(false);
            xAxis.setVisible(false); yAxis.setVisible(false); zAxis.setVisible(false);
            xText.setVisible(false); yText.setVisible(false); zText.setVisible(false);
            break;
          case VisualizationHints.DECORATION_CUBE :
            for (int i=0,n=boxSides.length; i<n; i++) boxSides[i].setVisible(true);
            xAxis.setVisible(false); yAxis.setVisible(false); zAxis.setVisible(false);
            xText.setVisible(false); yText.setVisible(false); zText.setVisible(false);
            break;
          case VisualizationHints.DECORATION_AXES :
            for (int i=0,n=boxSides.length; i<n; i++) boxSides[i].setVisible(false);
            xAxis.setVisible(true); yAxis.setVisible(true); zAxis.setVisible(true);
            xText.setVisible(true); yText.setVisible(true); zText.setVisible(true);
            break;
         }
         break;
      case VisualizationHints.HINT_ALPHA :
        cosAlpha=Math.cos(visHints.getAlpha());
        sinAlpha=Math.sin(visHints.getAlpha());
        needsToRecompute = true;
        break;
      case VisualizationHints.HINT_BETA  :
        cosBeta=Math.cos(visHints.getBeta());
        sinBeta=Math.sin(visHints.getBeta());
        needsToRecompute = true;
        break;
      case VisualizationHints.HINT_ALPHA_AND_BETA :
        cosAlpha=Math.cos(visHints.getAlpha()); sinAlpha=Math.sin(visHints.getAlpha());
        cosBeta =Math.cos(visHints.getBeta());  sinBeta =Math.sin(visHints.getBeta());
        needsToRecompute = true;
        break;
      case VisualizationHints.HINT_PAN:
      case VisualizationHints.HINT_ZOOM:
        needsToRecompute = true;
        break;
    }
    if (updateStrategy==UPDATE_AUTOMATIC) updatePanel();
  }

  /**
   * Converts a 3D point of the scene into a 2D point of the screen. It can
   * compute simultaneously the equivalent in screen coordinates of a distance in
   * the scene. Finally, it also provides a number measuring the relative
   * distance of the point to the useTODr's viewpoint.
   * distance = 1.0 means at the center of the scene,
   * distance > 1.0 means farther than the center of the scene,
   * distance < 1.0 means closer than the center of the scene,
   * @param coordinate The coordinates of the point of the scene
   * <itemize>
   * <li>If the input array has length 3, it is considered a single 3D point x,y,z
   * <li>If the input array has length 6, it is considered a 3D point plus a 3D vector x,y,z, dx,dy,dz
   * </itemize>
   * The input coordinates are not modified.
   * @param pixel a place-holder for the coordinates of the point of the screen
   * <itemize>
   * <li>If the input array had length 3, it returns a,b and the distance
   * <li>If the input array had length 6, it returns a,b, da,db and the distance
   * </itemize>
   * @return The coordinates of the point of the screen
   */
  double[] project (double[] coordinate, double[] pixel) {
    double x = coordinate[0] - centerX;
    double y = coordinate[1] - centerY;
    double z = coordinate[2] - centerZ;;
    double xprime, yprime, zprime, factor;
    switch (visHints.getDisplayMode()) {
      case VisualizationHints.DISPLAY_PLANAR_XY : xprime = z; yprime = x; zprime = y; factor = 1.8; break;
      case VisualizationHints.DISPLAY_PLANAR_XZ : xprime = y; yprime = x; zprime = z; factor = 1.8; break;
      case VisualizationHints.DISPLAY_PLANAR_YZ : xprime = x; yprime = y; zprime = z; factor = 1.8; break;
      case VisualizationHints.DISPLAY_NO_PERSPECTIVE :
        xprime =  x*cosAlpha + y*sinAlpha;
        yprime = -x*sinAlpha + y*cosAlpha;
        zprime = -xprime*sinBeta + z*cosBeta;
        xprime =  xprime*cosBeta + z*sinBeta;
        factor = 1.3;
        break;
      default :
      case VisualizationHints.DISPLAY_PERSPECTIVE :
        xprime =  x*cosAlpha + y*sinAlpha;
        yprime = -x*sinAlpha + y*cosAlpha;
        zprime = -xprime*sinBeta + z*cosBeta;
        xprime =  xprime*cosBeta + z*sinBeta;
        double aux = viewToCenter-xprime;
        if (Math.abs(aux)<1.0e-2) aux = 1.0e-2;  // This is to avoid division by zero
        factor = viewToPlane /aux; // (viewToCenter-xprime);
        break;
    }
    pixel[0] = acenter + yprime*factor*aconstant;
    pixel[1] = bcenter - zprime*factor*bconstant;
    if (coordinate.length==6) { // Input is x,y,z,dx,dy,dz
      switch (visHints.getDisplayMode()) {
        case VisualizationHints.DISPLAY_PLANAR_XY : // dx,dy are used
          pixel[2] = coordinate[3]*factor*aconstant;
          pixel[3] = coordinate[4]*factor*bconstant;
          break;
        case VisualizationHints.DISPLAY_PLANAR_XZ : // dx,dz are used
          pixel[2] = coordinate[3]*factor*aconstant;
          pixel[3] = coordinate[5]*factor*bconstant;
          break;
        case VisualizationHints.DISPLAY_PLANAR_YZ : // dy,dz are used
          pixel[2] = coordinate[4]*factor*aconstant;
          pixel[3] = coordinate[5]*factor*bconstant;
          break;
        default : /* 3D */ // max(dx,dy) and dz are used
          pixel[2] = Math.max(coordinate[3],coordinate[4])*factor*aconstant;
          pixel[3] = coordinate[5]*factor*bconstant;
          break;
      }
      pixel[4] = (viewToCenter-xprime)/viewToCenter;  // A number reporting about the distance to us
    }
    else { // Input is x,y,z
      pixel[2] = (viewToCenter-xprime)/viewToCenter;  // A number reporting about the distance to us
    }
    return pixel;
  }

  private float[] crc = new float[4]; // Stands for ColorRGBComponent

  /**
   * Computes the display color of a given drawable3D based on its original color and its depth.
   * Transparency of the original color is not affected.
   * @param _aColor the original color
   * @param _depth the depth value of the color
   */
  Color projectColor (Color _aColor, double _depth) {
    if (!visHints.isUseColorDepth()) return _aColor;
//    if      (_depth<0.9) return _aColor.brighter().brighter();
//    else if (_depth>1.1) return _aColor.darker().darker();
//    else return _aColor;
    try {
      _aColor.getRGBComponents(crc);
      // Do not affect transparency
      for (int i=0; i<3; i++) { crc[i] /= _depth; crc[i] = (float) Math.max(Math.min(crc[i],1.0),0.0); }
      return new Color(crc[0],crc[1],crc[2],crc[3]);
    } catch (Exception _exc)  { return _aColor; }
  }

  private void computeConstants () {
    int width  = this.getWidth();
    int height = this.getHeight();
    acenter = width/2 +visHints.getPanA();
    bcenter = height/2+visHints.getPanB();
    if (squareAspect) width = height = Math.min(width,height);
    double dx = xmax-xmin, dy = ymax-ymin, dz = zmax-zmin;
    double maxSpace;
    switch (visHints.getDisplayMode()) {
      case VisualizationHints.DISPLAY_PLANAR_XY : maxSpace = Math.max(dx,dy); break;
      case VisualizationHints.DISPLAY_PLANAR_XZ : maxSpace = Math.max(dx,dz); break;
      case VisualizationHints.DISPLAY_PLANAR_YZ : maxSpace = Math.max(dy,dz); break;
      default : /* 3D */                          maxSpace = Math.max(Math.max(dx,dy),dz); break;
    }
    centerX = (xmax+xmin)/2.0;
    centerY = (ymax+ymin)/2.0;
    centerZ = (zmax+zmin)/2.0;
    aconstant = 0.5*visHints.getZoom()*width/maxSpace;
    bconstant = 0.5*visHints.getZoom()*height/maxSpace;
    viewToPlane = ratioToPlane*maxSpace;
    viewToCenter = ratioToCenter*maxSpace;
    resetDecoration(dx,dy,dz);
    reportTheNeedToProject();
    needsToRecompute = false;
  }

  private void reportTheNeedToProject() {
    Iterator  it = getElements().iterator();
    while (it.hasNext()) ((Element) it.next()).setNeedToProject(true);
  }

  private void resetDecoration (double _dx, double _dy, double _dz) {
//    if (boxSides==null || boxSides[0]==null) return;
    boxSides[ 0].setXYZ(xmin,ymin,zmin); boxSides[ 0].setSizeXYZ(_dx,0.0,0.0);
    boxSides[ 1].setXYZ(xmax,ymin,zmin); boxSides[ 1].setSizeXYZ(0.0,_dy,0.0);
    boxSides[ 2].setXYZ(xmin,ymax,zmin); boxSides[ 2].setSizeXYZ(_dx,0.0,0.0);
    boxSides[ 3].setXYZ(xmin,ymin,zmin); boxSides[ 3].setSizeXYZ(0.0,_dy,0.0);
    boxSides[ 4].setXYZ(xmin,ymin,zmax); boxSides[ 4].setSizeXYZ(_dx,0.0,0.0);
    boxSides[ 5].setXYZ(xmax,ymin,zmax); boxSides[ 5].setSizeXYZ(0.0,_dy,0.0);
    boxSides[ 6].setXYZ(xmin,ymax,zmax); boxSides[ 6].setSizeXYZ(_dx,0.0,0.0);
    boxSides[ 7].setXYZ(xmin,ymin,zmax); boxSides[ 7].setSizeXYZ(0.0,_dy,0.0);
    boxSides[ 8].setXYZ(xmin,ymin,zmin); boxSides[ 8].setSizeXYZ(0.0,0.0,_dz);
    boxSides[ 9].setXYZ(xmax,ymin,zmin); boxSides[ 9].setSizeXYZ(0.0,0.0,_dz);
    boxSides[10].setXYZ(xmax,ymax,zmin); boxSides[10].setSizeXYZ(0.0,0.0,_dz);
    boxSides[11].setXYZ(xmin,ymax,zmin); boxSides[11].setSizeXYZ(0.0,0.0,_dz);
    xAxis.setXYZ(xmin,ymin,zmin); xAxis.setSizeXYZ(_dx,0.0,0.0); xText.setXYZ(xmax+_dx*0.02,ymin,zmin);
    yAxis.setXYZ(xmin,ymin,zmin); yAxis.setSizeXYZ(0.0,_dy,0.0); yText.setXYZ(xmin,ymax+_dy*0.02,zmin);
    zAxis.setXYZ(xmin,ymin,zmin); zAxis.setSizeXYZ(0.0,0.0,_dz); zText.setXYZ(xmin,ymin,zmax+_dz*0.02);
  }

// ----------------------------------------------------
// Interaction
// ----------------------------------------------------

  // Variables for interaction
  private int lastX=0, lastY=0;

  // This is so that the panel accepts KeyEvents
  public boolean isFocusable () { return true; }

  private void setMouseCursor(Cursor cursor) {
     Container c = getTopLevelAncestor();
     setCursor(cursor);
     if(c!=null) {
        c.setCursor(cursor);
     }
  }

  private void mouseDraggedComputations(java.awt.event.MouseEvent e) {
    if (e.isControlDown ()) { // Panning
      visHints.setPan (visHints.getPanA() + (e.getX()-lastX), visHints.getPanB() + (e.getY()-lastY));
    }
    else if (e.isShiftDown ()) { // Zooming
      visHints.setZoom (visHints.getZoom() - (e.getY()-lastY)*0.01);
    }
    else if (visHints.getDisplayMode()>=VisualizationHints.DISPLAY_3D && !e.isAltDown()) { // Rotating (in 3D)
      visHints.setAlphaAndBeta(visHints.getAlpha() - (e.getX()-lastX)*0.01, visHints.getBeta()  + (e.getY()-lastY)*0.01);
    }
  }

  private void resetInteraction () { repaint(); }

  /**
   * The inner class that will handle all mouse related events.
   */
  private class IADMouseController extends MouseController {

    public void mousePressed(MouseEvent _evt) {
      requestFocus();
      if (visHints.isAllowQuickRedraw() && ((_evt.getModifiers()&InputEvent.BUTTON1_MASK)!=0)) quickRedrawOn = true;
      else quickRedrawOn = false;
      lastX = _evt.getX(); lastY = _evt.getY();
      resetInteraction ();
      return;
    }

    public void mouseDragged(MouseEvent _evt) {
      mouseDraggedComputations(_evt);
      lastX = _evt.getX(); lastY = _evt.getY();
      repaint ();
    }

    public void mouseReleased(MouseEvent e) {
      quickRedrawOn = false;
      resetInteraction();
      setMouseCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
    }

    public void mouseEntered(MouseEvent e) {
      setMouseCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
    }

    public void mouseExited(MouseEvent e) {
      setMouseCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }

    public void mouseClicked(MouseEvent e) {
    }

    public void mouseMoved(MouseEvent e) {
    }
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

    /**
     * Returns an XML.ObjectLoader to save and load object data.
     * @return the XML.ObjectLoader
     */
    public static XML.ObjectLoader getLoader() { return new DrawingPanel3DLoader(); }

    static private class DrawingPanel3DLoader implements XML.ObjectLoader {

       public void saveObject(XMLControl control, Object obj) {
          DrawingPanel3D panel = (DrawingPanel3D) obj;
          control.setValue("preferred x min", panel.xmin);
          control.setValue("preferred x max", panel.xmax);
          control.setValue("preferred y min", panel.ymin);
          control.setValue("preferred y max", panel.ymax);
          control.setValue("preferred z min", panel.zmin);
          control.setValue("preferred z max", panel.zmax);
          control.setValue("update strategy", panel.updateStrategy);
          control.setValue("visualization hints", panel.visHints);
          control.setValue("elements", panel.getElements());
       }

       public Object createObject(XMLControl control) { return new DrawingPanel3D (); }

       /**
        * Loads a DrawingPanel with data from an XMLControl.
        *
        * @param control the control
        * @param obj the object
        * @return the loaded object
        */
       public Object loadObject(XMLControl control, Object obj) {
          DrawingPanel3D panel = (DrawingPanel3D) obj;
          panel.setUpdateStrategy(UPDATE_MANUAL); // Do not update while reading
          double minX = control.getDouble("preferred x min");
          double maxX = control.getDouble("preferred x max");
          double minY = control.getDouble("preferred y min");
          double maxY = control.getDouble("preferred y max");
          double minZ = control.getDouble("preferred z min");
          double maxZ = control.getDouble("preferred z max");
          panel.setPreferredMinMax(minX, maxX, minY, maxY, minZ, maxZ);
          VisualizationHints hints = (VisualizationHints) control.getObject("visualization hints");
          hints.setPanel(panel);
          panel.visHints = hints;
          panel.hintChanged(VisualizationHints.HINT_DECORATION_TYPE);
          panel.hintChanged(VisualizationHints.HINT_ALPHA_AND_BETA); // This one implies PAN and ZOOM
          Collection elements = (Collection) control.getObject("elements");
          if (elements!=null) {
            panel.removeAllElements();
            Iterator it = elements.iterator();
            while (it.hasNext()) panel.addElement((Element) it.next());
          }
          // I postpone it to the end because setUpdateStrategy(UPDATE_AUTOMATIC) invokes updatePanel()
          panel.setUpdateStrategy(control.getInt("update strategy"));
          panel.updatePanel();
          return obj;
       }
    } // End of static class DrawingPanel3DLoader

}


