/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.display.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.awt.print.*;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.*;
import java.rmi.RemoteException;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.tools.*;

/**
 * DrawingFrame: a frame that contains a generic drawing panel.
 * @author     Francisco Esquembre
 * @author     Adapted from Wolfgang Christian
 * @version    March 2005
 */
public class DrawingFrame3D extends OSPFrame
                            implements ClipboardOwner, org.opensourcephysics.display3d.core.DrawingFrame3D {
  protected JMenu fileMenu, editMenu;
  protected JMenuItem copyItem, pasteItem, replaceItem;

  protected JMenu visualMenu, displayMenu, decorationMenu;
  protected JMenuItem displayPerspectiveItem, displayNoPerspectiveItem, displayXYItem, displayXZItem, displayYZItem;
  protected JMenuItem decorationCubeItem, decorationNoneItem, decorationAxesItem;
  protected JMenuItem zoomToFitItem;

  protected JMenuBar menuBar = new JMenuBar();
  protected org.opensourcephysics.display3d.core.DrawingPanel3D drawingPanel;
  protected final static int MENU_SHORTCUT_KEY_MASK = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();

  /**
   *  Default DrawingFrame constructor
   */
  public DrawingFrame3D() {
    this("Drawing Frame", null);
  }

  /**
   *  DrawingFrame constructor specifying the DrawingPanel that will be placed
   *  in the center of the content pane.
   * @param  drawingPanel
   */
  public DrawingFrame3D(DrawingPanel3D drawingPanel) {
    this("Drawing Frame", drawingPanel);
  }

  /**
   *  DrawingFrame constructor specifying the title and the DrawingPanel that
   *  will be placed in the center of the content pane.
   *
   * @param  title
   * @param  _drawingPanel
   */
  public DrawingFrame3D(String title, DrawingPanel3D _drawingPanel) {
    super(title);
    drawingPanel = _drawingPanel;
    if (drawingPanel != null) getContentPane().add((JPanel) drawingPanel, BorderLayout.CENTER);
    pack();
    if (!OSPFrame.appletMode) createMenuBar();
  }

  /**
   * Renders the drawing panel if the frame is showing and not iconified.
   */
  public void render() {
    if (isIconified() ||  !isShowing()) return;
    if (drawingPanel!=null) drawingPanel.updatePanel();
    else repaint();
  }

  /**
   *  Gets the drawing panel.
   *
   * @return    the drawingPanel
   */
  public org.opensourcephysics.display3d.core.DrawingPanel3D getDrawingPanel3D() { return drawingPanel; }

  /**
   *  Adds the drawing panel to the the frame. The panel is added to the center
   *  of the frame's content pane.
   *
   * @param  _drawingPanel
   */
  public void setDrawingPanel3D(org.opensourcephysics.display3d.core.DrawingPanel3D _drawingPanel) {
    if(drawingPanel!=null) {  // remove the old drawing panel.
      getContentPane().remove((JPanel)drawingPanel);
    }
    drawingPanel = _drawingPanel;
    if(drawingPanel!=null) {
      getContentPane().add((JPanel)drawingPanel, BorderLayout.CENTER);
    }
    pack();
  }

  /**
   * Getting the pointer to the real JFrame in it
   * @return JFrame
   */
  public javax.swing.JFrame getJFrame() { return this; }

  /**
   *  This is a hack to fix a bug when the reload button is pressed in browsers
   *  running JDK 1.4.
   *
   * @param  g
   */
  public void paint(Graphics g) {
    if(!appletMode) {
      super.paint(g);
      return;
    }
    try {
      super.paint(g);
    } catch(Exception ex) {
      System.err.println("OSPFrame paint error: "+ex.toString());
      System.err.println("Title: "+this.getTitle());
    }
  }

  /**
   * Enables the paste edit menu item.
   * @param enable boolean
   */
  public void setEnabledPaste(boolean enable) {
    pasteItem.setEnabled(enable);
  }

  /**
   * Pastes drawables found in the specified xml control.
   *
   * @param control the xml control
   */
  protected void pasteAction(XMLControlElement control) {
    // get Drawables using an xml tree chooser
    XMLTreeChooser chooser = new XMLTreeChooser("Select Drawables", "Select one or more drawables.", this);
    java.util.List props = chooser.choose(control, Drawable.class);
    if(!props.isEmpty()) {
      Iterator it = props.iterator();
      while(it.hasNext()) {
        XMLControl prop = (XMLControl) it.next();
        Element element = (Element) prop.loadObject(null);
        drawingPanel.addElement(element);
      }
    }
    drawingPanel.updatePanel();
  }

  /**
   * Enables the replace edit menu item.
   * @param enable boolean
   */
  public void setEnabledReplace(boolean enable) {
    replaceItem.setEnabled(enable);
  }

  /**
   * Replaces the drawables with the drawables found in the specified XML control.
   * @param control XMLControlElement
   */
  public void replaceAction(XMLControlElement control) {
    drawingPanel.removeAllElements();
    pasteAction(control);
  }

  /**
   * Copies objects found in the specified xml control.
   *
   * @param control the xml control
   */
  protected void copyAction(XMLControlElement control) {
    StringSelection data = new StringSelection(control.toXML());
    Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    clipboard.setContents(data, this);
  }

  /**
   * Implementation of ClipboardOwner interface.
   *
   * Override this method to receive notification that data copied to the clipboard has changed.
   *
   * @param clipboard Clipboard
   * @param contents Transferable
   */
  public void lostOwnership(Clipboard clipboard, Transferable contents){
   }

  /**
   * Enables the copy edit menu item.
   * @param enable boolean
   */
  public void setEnabledCopy(boolean enable) {
    copyItem.setEnabled(enable);
  }

  /**
   * Creates a standard DrawingFrame menu bar and adds it to the frame.
   */
  private void createMenuBar() {
    fileMenu = new JMenu("File");
    JMenuItem printItem = new JMenuItem("Print...");
    printItem.setAccelerator(KeyStroke.getKeyStroke('P', MENU_SHORTCUT_KEY_MASK));
    printItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        PrinterJob printerJob = PrinterJob.getPrinterJob();
        printerJob.setPrintable((Printable)drawingPanel);
        if(printerJob.printDialog()) {
          try {
            printerJob.print();
          } catch(PrinterException pe) {
            JOptionPane.showMessageDialog(DrawingFrame3D.this,
                                          "A printing error occurred. Please try again.", "Error", JOptionPane.ERROR_MESSAGE);
          }
        }
      }
    });
    JMenuItem saveXMLItem = new JMenuItem("Save XML...");
    saveXMLItem.setAccelerator(KeyStroke.getKeyStroke('S', MENU_SHORTCUT_KEY_MASK));
    saveXMLItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        saveXML();
      }
    });

    JMenuItem exportItem = new JMenuItem("Export...");
    exportItem.setAccelerator(KeyStroke.getKeyStroke('E', MENU_SHORTCUT_KEY_MASK));
    exportItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          ExportTool.getTool().send(new LocalJob(drawingPanel), null);
        }
        catch (RemoteException ex) {}
      }
    });

    JMenuItem saveAsPSItem = new JMenuItem("Save As PS...");
    saveAsPSItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        GUIUtils.saveAsPS((Printable)drawingPanel, DrawingFrame3D.this);
      }
    });
    JMenuItem inspectItem = new JMenuItem("Inspect");
    inspectItem.setAccelerator(KeyStroke.getKeyStroke('I', MENU_SHORTCUT_KEY_MASK));
    inspectItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        inspectXML();  // cannot use a static method here because of run-time binding
      }
    });
    fileMenu.add(printItem);
    fileMenu.add(saveXMLItem);
    fileMenu.add(exportItem);
    fileMenu.add(saveAsPSItem);
    fileMenu.add(inspectItem);
    menuBar.add(fileMenu);

    editMenu = new JMenu("Edit");
    menuBar.add(editMenu);
    copyItem = new JMenuItem("Copy");
    copyItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        XMLControlElement control = new XMLControlElement(DrawingFrame3D.this);
        control.saveObject(null);
        copyAction(control);
      }
    });
    editMenu.add(copyItem);
    pasteItem = new JMenuItem("Paste");
    pasteItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
          Transferable data = clipboard.getContents(null);
          XMLControlElement control = new XMLControlElement();
          control.readXML((String) data.getTransferData(DataFlavor.stringFlavor));
          pasteAction(control);
        } catch(UnsupportedFlavorException ex) {}
        catch(IOException ex) {}
        catch(HeadlessException ex) {}
      }
    });
    pasteItem.setEnabled(false);  // not supported yet
    editMenu.add(pasteItem);
    replaceItem = new JMenuItem("Replace");
    replaceItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
          Transferable data = clipboard.getContents(null);
          XMLControlElement control = new XMLControlElement();
          control.readXML((String) data.getTransferData(DataFlavor.stringFlavor));
          replaceAction(control);
        } catch(UnsupportedFlavorException ex) {}
        catch(IOException ex) {}
        catch(HeadlessException ex) {}
      }
    });
    replaceItem.setEnabled(false);  // not supported yet
    editMenu.add(replaceItem);
    setJMenuBar(menuBar);

    displayMenu = new JMenu("Display mode");
    displayPerspectiveItem = new JMenuItem("3D Perspective");
    displayPerspectiveItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDisplayMode(VisualizationHints.DISPLAY_PERSPECTIVE);
          drawingPanel.updatePanel();
        }
      }
    });
    displayMenu.add(displayPerspectiveItem);
    displayNoPerspectiveItem = new JMenuItem("3D No perspective");
    displayNoPerspectiveItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDisplayMode(VisualizationHints.DISPLAY_NO_PERSPECTIVE);
          drawingPanel.updatePanel();
        }
      }
    });
    displayMenu.add(displayNoPerspectiveItem);
    displayXYItem = new JMenuItem("Planar XY");
    displayXYItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDisplayMode(VisualizationHints.DISPLAY_PLANAR_XY);
          drawingPanel.updatePanel();
        }
      }
    });
    displayMenu.add(displayXYItem);
    displayXZItem = new JMenuItem("Planar XZ");
    displayXZItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDisplayMode(VisualizationHints.DISPLAY_PLANAR_XZ);
          drawingPanel.updatePanel();
        }
      }
    });
    displayMenu.add(displayXZItem);
    displayYZItem = new JMenuItem("Planar YZ");
    displayYZItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDisplayMode(VisualizationHints.DISPLAY_PLANAR_YZ);
          drawingPanel.updatePanel();
        }
      }
    });
    displayMenu.add(displayYZItem);

    decorationMenu = new JMenu("Decoration");
    decorationNoneItem = new JMenuItem("No decoration");
    decorationNoneItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDecorationType(VisualizationHints.DECORATION_NONE);
          drawingPanel.updatePanel();
        }
      }
    });
    decorationMenu.add(decorationNoneItem);
    decorationCubeItem = new JMenuItem("Cube");
    decorationCubeItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDecorationType(VisualizationHints.DECORATION_CUBE);
          drawingPanel.updatePanel();
        }
      }
    });
    decorationMenu.add(decorationCubeItem);
    decorationAxesItem = new JMenuItem("Axes");
    decorationAxesItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.getVisualizationHints().setDecorationType(VisualizationHints.DECORATION_AXES);
          drawingPanel.updatePanel();
        }
      }
    });
    decorationMenu.add(decorationAxesItem);

    zoomToFitItem = new JMenuItem("Zoom to fit");
    zoomToFitItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (drawingPanel != null) {
          drawingPanel.zoomToFit();
          drawingPanel.updatePanel();
        }
      }
    });

    visualMenu = new JMenu("Visual");
    visualMenu.add(displayMenu);
    visualMenu.add(decorationMenu);
    visualMenu.add(zoomToFitItem);
    menuBar.add(visualMenu);

    //loadToolsMenu();
    JMenu helpMenu = new JMenu("Help");
    menuBar.add(helpMenu);
    JMenuItem aboutItem = new JMenuItem("About...");
    aboutItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        ControlFrame.showAboutDialog(DrawingFrame3D.this);
      }
    });
    helpMenu.add(aboutItem);
  }

  /**
   * Gets a menu with the given name from the menu bar.  Returns null if menu item does not exist.
   *
   * @param menuName String
   * @return JMenu
   */
  public JMenu getMenuItem(String menuName){
     menuName=menuName.trim();
     JMenu menu = null;
     for (int i = 0; i<menuBar.getMenuCount(); i++){
        JMenu next = menuBar.getMenu(i);
        if (next.getText().equals(menuName)){
           menu = next;
           break;
        }
     }
     return menu;
  }

  /**
   * Removes a menu with the given name from the menu bar and returns the removed item.
   * Returns null if menu item does not exist.
   *
   * @param menuName String
   * @return JMenu
   */
  public JMenu removeMenuItem(String menuName){
     menuName = menuName.trim();
     JMenu menu = null;
     for (int i = 0; i<menuBar.getMenuCount(); i++){
        JMenu next = menuBar.getMenu(i);
        if (next.getText().equals(menuName)){
           menu = next;
           menuBar.remove(i);
           break;
        }
     }
     return menu;
  }


  /**
   * Inspects the drawing frame by using an xml document tree.
   */
  public void inspectXML() {
    XMLControlElement xml = null;
    try {
      // if drawingPanel provides an xml loader, inspect the drawingPanel
      Method method = drawingPanel.getClass().getMethod("getLoader");
      if((method!=null)&&Modifier.isStatic(method.getModifiers())) {
        xml = new XMLControlElement(drawingPanel);
      }
    } catch(NoSuchMethodException ex) {
      // this drawing panel cannot be inspected
      return;
    }
    // display a TreePanel in a modal dialog
    XMLTreePanel treePanel = new XMLTreePanel(xml);
    JDialog dialog = new JDialog((java.awt.Frame) null, true);
    dialog.setContentPane(treePanel);
    dialog.setSize(new Dimension(600, 300));
    dialog.setVisible(true);
  }

  public void saveXML() {
    JFileChooser chooser = getChooser();
    int result = chooser.showSaveDialog(null);
    if(result==JFileChooser.APPROVE_OPTION) {
      File file = chooser.getSelectedFile();
      // check to see if file already exists
      if(file.exists()) {
        int selected = JOptionPane.showConfirmDialog(null, "Replace existing "+file.getName()+"?", "Replace File", JOptionPane.YES_NO_CANCEL_OPTION);
        if(selected!=JOptionPane.YES_OPTION) {
          return;
        }
      }
      String fileName = XML.getRelativePath(file.getAbsolutePath());
      if((fileName==null)||fileName.trim().equals("")) {
        return;
      }
      int i = fileName.toLowerCase().lastIndexOf(".xml");
      if(i!=fileName.length()-4) {
        fileName += ".xml";
      }
      try {
        // if drawingPanel provides an xml loader, save the drawingPanel
        Method method = drawingPanel.getClass().getMethod("getLoader");
        if((method!=null)&&Modifier.isStatic(method.getModifiers())) {
          XMLControl xml = new XMLControlElement(drawingPanel);
          xml.write(fileName);
        }
      } catch(NoSuchMethodException ex) {
        // this drawingPanel cannot be saved
        return;
      }
    }
  }

  /**
   * Returns an XML.ObjectLoader to save and load data for this program.
   *
   * @return the object loader
   */
  public static XML.ObjectLoader getLoader() {
    return new DrawingFrame3DLoader();
  }

   static class DrawingFrame3DLoader implements XML.ObjectLoader {

    /**
     * createObject
     *
     * @param control XMLControl
     * @return Object
     */
    public Object createObject(XMLControl control) {
      DrawingFrame frame = new DrawingFrame();
      frame.setTitle(control.getString("title"));
      frame.setLocation(control.getInt("location x"), control.getInt("location y"));
      frame.setSize(control.getInt("width"), control.getInt("height"));
      if(control.getBoolean("showing")) {
        frame.setVisible(true);
      }
      return frame;
    }

    /**
     * Save data object's data in the control.
     *
     * @param control XMLControl
     * @param obj Object
     */
    public void saveObject(XMLControl control, Object obj) {
      DrawingFrame3D frame = (DrawingFrame3D) obj;
      control.setValue("title", frame.getTitle());
      control.setValue("showing", frame.isShowing());
      control.setValue("location x", frame.getLocation().x);
      control.setValue("location y", frame.getLocation().y);
      control.setValue("width", frame.getSize().width);
      control.setValue("height", frame.getSize().height);
      control.setValue("drawing panel", frame.getDrawingPanel3D());
    }

    /**
     * Loads the object with data from the control.
     *
     * @param control XMLControl
     * @param obj Object
     * @return Object
     */
    public Object loadObject(XMLControl control, Object obj) {
      DrawingFrame3D frame = ((DrawingFrame3D) obj);
      DrawingPanel3D panel = (DrawingPanel3D) frame.getDrawingPanel3D();
      panel.removeAllElements();
      XMLControl panelControl = control.getChildControl("drawing panel");
      panelControl.loadObject(panel);
      panel.updatePanel();
      frame.setTitle(control.getString("title"));
      frame.setLocation(control.getInt("location x"), control.getInt("location y"));
      frame.setSize(control.getInt("width"), control.getInt("height"));
      if(control.getBoolean("showing")) {
        frame.setVisible(true);
      }
      return obj;
    }
  }
}
