/*
 * The org.opensourcephysics.display3d.core package provides the basic
 * interfaces and classes that conform a basic drawing framework for 3D scenes.
 *
 * Subpackages should implement these interfaces using any of the available
 * 3D rendering mechanisms.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.core;

/**
 * <p>Title: VisualizationHints</p>
 * <p>Description: Hints to a DrawingPanel3D about how it should look.
 * Hints can be ignored by the panel, depending on the implementation.</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public interface VisualizationHints {
  int DISPLAY_PLANAR  = 0;
  int DISPLAY_PLANAR_XY  = DISPLAY_PLANAR;
  int DISPLAY_PLANAR_XZ  = 1;
  int DISPLAY_PLANAR_YZ  = 2;
  int DISPLAY_3D  = 10; // Thus displayMode<DISPLAY_3D implies any PLANAR mode
  int DISPLAY_PERSPECTIVE = DISPLAY_3D;
  int DISPLAY_NO_PERSPECTIVE = 12;

  int DECORATION_NONE = 0;
  int DECORATION_AXES = 1;
  int DECORATION_CUBE = 2;

  /**
   * The display mode for the panel. One of the following
   * <ul>
   *   <li><b>DISPLAY_PERSPECTIVE</b>: 3D view with perspective. The default.</li>
   *   <li><b>DISPLAY_NO_PERSPECTIVE</b>: 3D view with no perspective.</li>
   *   <li><b>DISPLAY_XY</b>: 2D view using X and Y coordinates.</li>
   *   <li><b>DISPLAY_XZ</b>: 2D view using X and Z coordinates.</li>
   *   <li><b>DISPLAY_YZ</b>: 2D view using Y and Z coordinates.</li>
   * </ul>
   * @param mode the desired value
   */
  public void setDisplayMode (int mode);
  public int getDisplayMode ();

  /**
   * Types of decoration displayed. One of the following
   * <ul>
   *   <li><b>DECORATION_NONE</b>: No decoration</li>
   *   <li><b>DECORATION_AXES</b>: Display labelled axes</li>
   *   <li><b>DECORATION_CUBE</b>: Display the bounding box</li>
   * </ul>
   * @param type the desired value
   */
  public void setDecorationType (int type);
  public int getDecorationType ();

  /**
   * Whether the panel should try to remove hidden lines
   * @param remove the desired value
   */
  public void setRemoveHiddenLines (boolean remove);
  public boolean isRemoveHiddenLines ();

  /**
   * Whether the panel can draw quickly when it is dragged for a
   * new view point
   * @param allow the desired value
   */
  public void setAllowQuickRedraw (boolean allow);
  public boolean isAllowQuickRedraw ();

  /**
   * Whether the panel should display far objects darker
   * @param useIt the desired value
   */
  public void setUseColorDepth (boolean useIt);
  public boolean isUseColorDepth ();

  /**
   * Set the displacement of the center of the scene (in pixels) with respect to
   * the center of the panel. Default is 0,0.
   * @param _panA The horizontal displacement
   * @param _panB The vertical displacement
   */
  public void setPan (int _panA, int _panB);
  public java.awt.Point getPan ();
  public int getPanA ();
  public int getPanB ();

  /**
   * Set the angle (in radians) to rotate the scene horizontally
   * before projecting. Default is 0.0.
   * @param _alpha the desired angle
   */
  public void setAlpha (double _alpha);
  public double getAlpha ();

  /**
   * Set the angle (in radians) to rotate the scene vertically
   * before projecting. Default is 0.0.
   * @param _beta the desired angle
   */
  public void setBeta (double _beta);
  public double getBeta ();

  /**
   * Set the angles (in radians) to rotate the scene horizontally and vertically before projecting
   * @param _alpha the desired horizontal angle
   * @param _beta the desired vertical angle
   */
  public void setAlphaAndBeta (double _alpha, double _beta);

  /**
   * Set the magnifying factor to apply to the scene. Default is 1.0.
   * @param _zoom the desired value
   */
  public void setZoom (double _zoom);
  public double getZoom ();

}
