/*
 * The org.opensourcephysics.display3d.core package provides the basic
 * interfaces and classes that conform a basic drawing framework for 3D scenes.
 *
 * Subpackages should implement these interfaces using any of the available
 * 3D rendering mechanisms.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.core;

import org.opensourcephysics.controls.*;

/**
 * <p>Title: Resolution</p>
 * <p>Description: A class that provides resolution style hints for 3D
 * Elements</p>
 * <p>A resolution can be of two different types: DIVISIONS or MAX_LENGTH.</p>
 * <ul>
 *   <li><b>DIVISIONS</b>: the resolution indicates how many subdivisions
 * should the element have, up to three of them: n1, n2, and n3. The precise
 * meaning of this is left to the element, but typically consists in the
 * number of divisions in each coordinate direction.</li>
 *   <li><b>MAX_LENGTH</b>: the resolution provides a maximum length
 * that each of the individual graphical pieces of the element can have.
 * The element can then automatically divide itself in smaller pieces,
 * if necessary.</li>
 * </ul>
 * @author Francisco Esquembre
 * @version March 2005
 * @see Style
 */
public class Resolution {
  static public final int DIVISIONS  = 0;
  static public final int MAX_LENGTH = 1;

  private int type=DIVISIONS;
  private double maxLength = 1.0;
  private int n1=1;
  private int n2=1;
  private int n3=1;

  /**
   * A constructor for a resolution of type MAX_LENGTH.
   * @param max The maximum length.
   */
  public Resolution  (double max) {
    this.type = MAX_LENGTH;
    this.maxLength = max;
  }

  /**
   * A constructor for a resolution of type DIVISIONS.
   * @param n1 int the first number of subdivisions
   * @param n2 int the second number of subdivisions
   * @param n3 int the thrid number of subdivisions
   */
  public Resolution (int n1, int n2, int n3) {
    this.type = DIVISIONS;
    this.n1 = n1;
    this.n2 = n2;
    this.n3 = n3;
  }

  final public int getType () { return this.type; }
  final public double getMaxLength () { return this.maxLength; }
  final public int getN1 () { return this.n1; }
  final public int getN2 () { return this.n2; }
  final public int getN3 () { return this.n3; }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  public static XML.ObjectLoader getLoader() {
    return new ResolutionLoader();
  }

  public static class ResolutionLoader extends XMLLoader {
    public void saveObject(XMLControl control, Object obj) {
      Resolution res = (Resolution) obj;
      control.setValue("type", res.type);
      control.setValue("max length", res.maxLength);
      control.setValue("n1", res.n1);
      control.setValue("n2", res.n2);
      control.setValue("n3", res.n3);
    }

    public Object createObject(XMLControl control) {
      return new Resolution (1,1,1);
    }

    public Object loadObject(XMLControl control, Object obj) {
      Resolution res = (Resolution) obj;
      res.type = control.getInt("type");
      res.maxLength = control.getDouble("max length");
      res.n1 = control.getInt("n1");
      res.n2 = control.getInt("n2");
      res.n3 = control.getInt("n3");
      return obj;
    }
  }

}
