/*
 * The org.opensourcephysics.display3d.core package provides the basic
 * interfaces and classes that conform a basic drawing framework for 3D scenes.
 *
 * Subpackages should implement these interfaces using any of the available
 * 3D rendering mechanisms.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.core;

/**
 * <p>Title: ElementTrail</p>
 * Description:<p>A trail of 3D pixels on the screen.</p>
 * This object is often used to show the path of a moving object.
 * @author Francisco Esquembre
 * @version March 2005
 */
public interface ElementTrail extends Element {

    /**
     * Adds a new (x,y,z) point to the trail.
     * @param x double
     * @param y double
     * @param z double
     */
    public void addPoint(double x, double y, double z);

    /**
     * Sets the maximum number of points for the trail.
     * Once the maximum is reached, adding a new point will cause
     * remotion of the first one. This is useful to keep trails
     * down to a reasonable size, since very long trails can slow
     * down the rendering (in certain implementations).
     * If the value is 0 (the default) the trail grows forever
     * without discarding old points.
     * @param maximum int
     */
    public void setMaximumPoints (int maximum);

    /**
     * Returns the maximum number of points allowed for the trail
     * @return int
     */
    public int getMaximumPoints ();

    /**
     * Sets the connected flag.
     * Successive points are connected by a segment if this flag is true.
     * Each point is marked as a colored pixel if the trail is not connected.
     * Setting it temporarily to false helps create discontinuous trails.
     * @param connected boolean
     */
    public void setConnected(boolean connected);

    /**
     * Gets the connected flag.
     * @param connected boolean
     * @see #setConnected(boolean)
     */
    public boolean isConnected(boolean connected);

    /**
     * Starts a new (x,y,z) trail segment by moving to a new point
     * without drawing. (Equivalent to setting the connected flag
     * to false and adding one singlepoint, then setting the flag
     * back to true.)
     * @param x double
     * @param y double
     * @param z double
     */
    public void moveToPoint(double x, double y, double z);

    /**
     * Clears all points from the trail.
     */
    public void clear();

}













