/*
 * The org.opensourcephysics.display3d.core package provides the basic
 * interfaces and classes that conform a basic drawing framework for 3D scenes.
 *
 * Subpackages should implement these interfaces using any of the available
 * 3D rendering mechanisms.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.core;

/**
 * <p>Title: Element</p>
 * <p>Description: A basic individual 3D element.</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public interface Element {

// ----------------------------------------
// Name of the element
// ----------------------------------------

  /**
   * Gives a name to the element.
   * Naming an element is optional, but the element may use its name
   * to identify itself in XML files, for instance.
   * @param name String
   */
  public void setName (String name);

  /**
   * Gets the name of the element
   * @return String the name
   */
  public String getName();

// ----------------------------------------
// Position of the element
// ----------------------------------------

  /**
   * Set the X coordinate of the element
   * @param x double
   */
  public void setX(double x);
  /**
   * Get the X coordinate of the element
   * @return double
   */
  public double getX();

  /**
   * Set the Y coordinate of the element
   * @param y double
   */
  public void setY(double y);
  /**
   * Get the Y coordinate of the element
   * @return double
   */
  public double getY();

  /**
   * Set the Z coordinate of the element
   * @param z double
   */
  public void setZ(double z);

  /**
   * Get the Z coordinate of the element
   * @return double
   */
  public double getZ();

  /**
   * Set the X, Y and Z coordinates of the element
   * @param x double
   * @param y double
   * @param z double
   */
  public void setXYZ(double x, double y, double z);

// ----------------------------------------
// Size of the element
// ----------------------------------------

  /**
   * Set the size along the X axis
   * @param sizeX double
   */
  public void setSizeX(double sizeX);

  /**
   * Get the size along the X axis
   * @return double
   */
  public double getSizeX();

  /**
   * Set the size along the Y axis
   * @param sizeY double
   */
  public void setSizeY(double sizeY);

  /**
   * Get the size along the Y axis
   * @return double
   */
  public double getSizeY();

  /**
   * Set the size along the Z axis
   * @param sizeZ double
   */
  public void setSizeZ(double sizeZ);

  /**
   * Get the size along the Z axis
   * @return double
   */
  public double getSizeZ();

  /**
   * Set the size along the X, Y and Z axes
   * @param sizeX double
   * @param sizeY double
   * @param sizeZ double
   */
  public void setSizeXYZ(double sizeX, double sizeY, double sizeZ);

// -------------------------------------
// Visibility and style
// -------------------------------------

  /**
   * Sets the visibility of the element
   * @param _visible boolean
   */
  public void setVisible (boolean _visible);
  /**
   * Whether the element is visible
   * @return boolean
   */
  public boolean isVisible ();

  /**
   * Gets the style of the element
   * @return Style
   * @see Style
   */
  public Style getStyle ();

// ----------------------------------------
// Transformation of the element
// ----------------------------------------

  /**
   * Sets the internal transformation of the element, that is, the
   * transformation that converts the standard XYZ axes to the body's
   * internal reference axes.
   * The transformation is copied and cannot be accessed by users
   * directy. This implies that changing the original transformation
   * has no effect on the element unless a new setTransformation() is invoked.
   * The transformation uses the body's position as its origin.
   * @param transformation the new transformation
   * @see org.opensourcephysics.numerics.Transformation
   */
  public void setTransformation (org.opensourcephysics.numerics.Transformation transformation);

  /**
   * This method converts a double[3] vector from the body's frame to
   * the space's frame.
   * @param vector the vector to be converted
   */
  public void toSpaceFrame (double[] vector);

  /**
   * This method converts a double[3] vector from the space's frame to
   * the body's frame. </p>
   * This only works properly if the internal transformation is not set
   * (i.e. it is the identity) or if it is invertible.
   * Otherwise, a call to this method will throw an
   * UnsupportedOperationException exception.
   * @param vector the vector to be converted
   */
  public void toBodyFrame (double[] vector) throws UnsupportedOperationException;

}

