/*
 * The org.opensourcephysics.display3d.core package provides the basic
 * interfaces and classes that conform a basic drawing framework for 3D scenes.
 *
 * Subpackages should implement these interfaces using any of the available
 * 3D rendering mechanisms.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.core;

/**
 * <p>Title: DrawingPanel3D</p>
 * <p>Description: DrawingPanel3D is the basic 3D drawing panel</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public interface DrawingPanel3D {
  public static int UPDATE_AUTOMATIC = 0;
  public static int UPDATE_MANUAL = 1;

  /**
   * Getting the pointer to the real JPanel in it
   * @return JFrame
   */
  public javax.swing.JPanel getJPanel();

// ---------------------------------
// Customization of the panel
// ---------------------------------
  /**
   * Sets the preferred extrema for the panel
   * @param minX double
   * @param maxX double
   * @param minY double
   * @param maxY double
   * @param minZ double
   * @param maxZ double
   */
  public void setPreferredMinMax (double minX, double maxX,
                                  double minY, double maxY,
                                  double minZ, double maxZ);
  /**
   * Gets the preferred minimum in the X coordinate
   * @return double
   */
  public double getPreferredMinX ();
  /**
   * Gets the preferred maximum in the X coordinate
   * @return double
   */
  public double getPreferredMaxX ();
  /**
   * Gets the preferred minimum in the Y coordinate
   * @return double
   */
  public double getPreferredMinY ();
  /**
   * Gets the preferred maximum in the Y coordinate
   * @return double
   */
  public double getPreferredMaxY ();
  /**
   * Gets the preferred minimum in the Z coordinate
   * @return double
   */
  public double getPreferredMinZ ();
  /**
   * Gets the preferred maximum in the Z coordinate
   * @return double
   */
  public double getPreferredMaxZ ();

  /**
   * Sets the preferred min and max in each dimension so that all
   * elements currently in the panel are visible.
   */
  public void zoomToFit();

  /**
   * Whether the panel should try to keep a square aspect.
   * Default value is true.
   * @param square boolean
   */
  public void setSquareAspect (boolean square);

  /**
   * Whether the panel tries to keep a square aspect.
   * @return boolean
   */
  public boolean isSquareAspect ();

  /**
   * Provides the list of visualization hints that the panel uses
   * to display the 3D scene
   * @return VisualizationHints
   * @see VisualizationHints
   */
  public VisualizationHints getVisualizationHints ();

  /**
   * This method refreshes the DrawingPanel3D.
   * @see #setUpdateStrategy()
   */
  public void updatePanel();

  /**
   * Sets the update strategy.
   * Accepted values may depend on the implementation,
   * but at least two are supported:
   * <ul>
   *   <li>DrawingPanel3D.UPDATE_AUTOMATIC: the panel updates itself whenever
   *       it changes.
   *   <li>DrawingPanel3D.UPDATE_MANUAL: the panel doesn't update itself
   *       whenever it changes. A call to updatePanel() is required to update
   *       the panel.
   * </ul>
   * @see #updatePanel()
   */
  public void setUpdateStrategy(int strategy);

  /**
   * Returns the update strategy
   * @return int
   */
  public int getUpdateStrategy ();

  /**
   * Adds an Element to this DrawingPanel3D.
   * @param element Element
   * @see Element
   */
  public void addElement(Element element);

  /**
   * Removes an Element from this DrawingPanel3D
   * @param element Element
   * @see Element
   */
  public void removeElement(Element element);

  /**
   * Removes all Elements from this DrawingPanel3D
   * @see Element
   */
  public void removeAllElements();

  /**
   * Gets the (cloned) list of Elements.
   * (Should be synchronized.)
   * @return cloned list
   */
  public java.util.ArrayList getElements();

}
