/*
* The org.opensourcephysics.display package contains components for rendering
* two-dimensional scalar and vector fields.
* Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;
import java.awt.Color;
import javax.swing.JFrame;
import org.opensourcephysics.display.Measurable;

/**
 * The Plot2D interface defines common methods for 2d-plotting such as a GridPlot,
 * a VectorPlot, or a ContourPlot.
 *
 * Data must be stored in a GridData object.
 *
 * @author     Wolfgang Christian
 * @created    May 28, 2003
 * @version    1.0
 */
public interface Plot2D extends Measurable {

   /**
    * Sets the data to new values.
    *
    * The grid is resized to fit the new data if needed.
    *
    * @param val an array of new values
    */
   public void setAll(Object val);

   /**
    * Sets the values and the scale.
    *
    * The grid is resized to fit the new data if needed.
    *
    * @param val array of new values
    * @param xmin double
    * @param xmax double
    * @param ymin double
    * @param ymax double
    */
   public void setAll(Object obj, double xmin, double xmax, double ymin, double ymax);

   /**
    * Sets the data storage to the given value.
    *
    * @param _pointdata
    */
   public void setGridData(GridData _griddata);

   /**
    * Gets the GridData object.
    * @return GridData
    */
   public GridData getGridData();

   /**
    * Gets the x coordinate for the given index.
    *
    * @param i int
    * @return double the x coordinate
    */
   public double indexToX(int i);

   /**
    * Gets the y coordinate for the given index.
    *
    * @param i int
    * @return double the y coordinate
    */
   public double indexToY(int i);

   /**
    * Gets closest index from the given x  world coordinate.
    *
    * @param x double the coordinate
    * @return int the index
    */
   public int xToIndex(double x);

   /**
    * Gets closest index from the given y  world coordinate.
    *
    * @param x double the coordinate
    * @return int the index
    */
   public int yToIndex(double y);

   /**
    * Gets the autoscale flag for z.
    *
    * @return boolean
    */
   public boolean isAutoscaleZ();

   /**
    * Gets the floor for scaling the z data.
    * @return double
    */
   public double getFloor();

   /**
    * Gets the ceiling for scaling the z data.
    * @return double
    */
   public double getCeiling();

   /**
    * Sets the autoscale flag and the floor and ceiling values for the colors.
    *
    * If autoscaling is true, then the min and max values of z are span the colors.
    *
    * If autoscaling is false, then floor and ceiling values limit the colors.
    * Values below min map to the first color; values above max map to the last color.
    *
    * @param isAutoscale
    * @param floor
    * @param ceil
    */
   public void setAutoscaleZ(boolean isAutoscale, double floor, double ceil);

   /**
    * Sets the floor and ceiling colors.
    *
    * @param floorColor
    * @param ceilColor
    */
   public void setFloorCeilColor(Color floorColor, Color ceilColor);

   /**
    * Sets the colors that will be used between the floor and ceiling values.
    *
    * @param colors
    */
   public void setColorPalette(Color[] colors);

   /**
    * Determines the palette type that will be used.
    * @param type
    */
   public void setPaletteType(int type);

   /**
    *  Sets the color for grid line boundaries
    *
    * @param  c
    */
   public void setGridLineColor(Color c);

   /**
    * Outlines the data grid's boundaries.
    *
    * @param showGridLines
    */
   public void setShowGridLines(boolean showGrid);

   /**
    * Shows how values map to colors.
    */
   public JFrame showLegend();

   /**
    * Sets the visibility of the plot.
    * Drawing will be disabled if visible is false.
    *
    * @param isVisible
    */
   public void setVisible(boolean isVisible);

   /**
    * Sets the indexes for the data components that will be plotted.
    *
    * Indexes determine the postion of the amplitude, phase, x-component, and y-component
    * data in the data array.  The amplitude index is usually the first index.
    *
    * @param n the sample-component indexes
    */
   public void setIndexes(int[] indexes);

   /**
    * Updates this object's state using new data values.
    *
    * Update should be invoked if the data in the PointData object changes or if the z scale
    * of the PointData object changes.
    *
    */
   public void update();
}
