/*
 * The org.opensourcephysics.display package contains components for rendering
 * two-dimensional scalar and vector fields.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;

import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.image.*;
import org.opensourcephysics.display.*;


/**
 * A IntegerRaster contains an array of integers where each integer representents an image pixel.
 *
 * Because the image created by a IntegerRaster cannot be resized, the image dimensions
 * are the same as the dimensions of the integer array.
 *
 * @author     Wolfgang Christian
 * @created    February 11, 2003
 * @version    1.0
 */

public class IntegerRaster extends MeasuredImage implements Dimensioned{
  public static int WHITE= 0xFFFFFF;
  WritableRaster raster;
  byte[][] rgbData;
  int nrow, ncol;
  boolean visible = true;
  Dimension dimension;

  /**
   * Constructs IntegerRaster with the given size.
   * @param _nrow the number of rows
   * @param _ncol the number of columns
   */
  public IntegerRaster(int _nrow, int _ncol) {
    nrow=_nrow;
    ncol=_ncol;
    dimension = new Dimension(ncol, nrow);
    int size=nrow*ncol;
    ComponentColorModel ccm = new ComponentColorModel(
    ColorSpace.getInstance(ColorSpace.CS_sRGB),
    new int[] {8,8,8},
    false,  // hasAlpha
    false,
    Transparency.OPAQUE,
    DataBuffer.TYPE_BYTE);
    BandedSampleModel csm = new BandedSampleModel(
        DataBuffer.TYPE_BYTE,
        ncol,
        nrow,
        ncol,
        new int[]{0,1,2},
        new int[]{0,0,0});

    rgbData=new byte[3][size];
    DataBuffer databuffer= new DataBufferByte(rgbData,size);
    WritableRaster raster= Raster.createWritableRaster(csm,databuffer, new Point(0,0) );
    image = new BufferedImage(ccm,raster,false,null);

    // col in x direction, row in y direction
    xmin=0;
    xmax=ncol;
    ymin=nrow;
    ymax=0;  // zero is on top
  }

  /**
   * Sets a block of lattice cells to new values.
   *
   * @param row_offset
   * @param col_offset
   * @param val
   */
  public void setBlock(int row_offset, int col_offset, int val[][]) {
    if ( (row_offset < 0) || (row_offset + val.length > nrow)) {
      throw new IllegalArgumentException(
          "Row index out of range in integer raster setBlock.");
    }
    if ( (col_offset < 0) || (col_offset + val[0].length > ncol)) {
      throw new IllegalArgumentException(
          "Column index out of range in integer raster setBlock.");
    }

    for (int rindex = row_offset, nr = val.length + row_offset; rindex < nr; rindex++) {
      for (int cindex = col_offset, nc = val[0].length + col_offset; cindex < nc; cindex++) {
        int index = rindex * ncol + cindex;
        int pixval = val[rindex - row_offset][cindex - col_offset];
        rgbData[0][index] = (byte) ( (pixval >> 16) & 0xFF); // red
        rgbData[1][index] = (byte) ( (pixval >> 8) & 0xFF); // green
        rgbData[2][index] = (byte) ( (pixval >> 0) & 0xFF); // blue
      }
    }
  }

  /**
   * Sets a row of lattice cells to new values.
   *
   * @param row
   * @param col_offset
   * @param val
   */
  public void setRow(int row, int col_offset, int val[]) {
    if ( (row < 0) || (row>= nrow)) {
      throw new IllegalArgumentException(
          "Row index out of range in integer raster setBlock.");
    }
    if ( (col_offset < 0) || (col_offset + val.length > ncol)) {
      throw new IllegalArgumentException(
          "Column index out of range in integer raster setBlock.");
    }
    for (int cindex = col_offset, nc = val.length + col_offset; cindex < nc; cindex++) {
      int index = row * ncol + cindex;
      int pixval = val[cindex - col_offset];
      rgbData[0][index] = (byte) ( (pixval >> 16) & 0xFF); // red
      rgbData[1][index] = (byte) ( (pixval >> 8) & 0xFF); // green
      rgbData[2][index] = (byte) ( (pixval >> 0) & 0xFF); // blue
    }
  }

  /**
 * Sets a column of lattice cells to new values.
 *
 * @param row_offset
 * @param col
 * @param val
 */
  public void setCol(int row_offset, int col, int val[]) {
    if ( (row_offset < 0) || (row_offset + val.length > nrow)) {
      throw new IllegalArgumentException(
          "Row index out of range in integer raster setBlock.");
    }
    if ( (col < 0) || (col >= ncol)) {
      throw new IllegalArgumentException(
          "Column index out of range in integer raster setBlock.");
    }
    for (int rindex = row_offset, nr = val.length + row_offset; rindex < nr; rindex++) {
      int index = rindex * ncol + col;
      int pixval = val[rindex - row_offset];
      rgbData[0][index] = (byte) ( (pixval >> 16) & 0xFF); // red
      rgbData[1][index] = (byte) ( (pixval >> 8) & 0xFF); // green
      rgbData[2][index] = (byte) ( (pixval >> 0) & 0xFF); // blue
    }
  }

  /**
   * Sets a lattice cell to a new value.
   */
  public void setCell(int _row, int _col, int val){
    int index=_row*ncol+_col;
    rgbData[0][index]=(byte)((val >> 16) & 0xFF); // red
    rgbData[1][index]=(byte)((val >> 8) & 0xFF);  // green
    rgbData[2][index]=(byte)((val >> 0) & 0xFF);  // blue

  }

  /**
   * Gets a lattice cell value.
   *
   * @return the cell value.
   */
  public int getCell(int _row, int _col){
    int index=_row*ncol+_col;
    return    (((int)rgbData[0][index] & 0xFF) << 16) |
              (((int)rgbData[1][index] & 0xFF) << 8)  |
              (((int)rgbData[2][index] & 0xFF) << 0);

  }

  /**
   * Gets the dimension of the lattice in pixel units.
   *
   * Because Java does not support scaling of packed rasters, the
   * image can only be translated.
   *
   * @param panel
   * @return the dimension
   */
  public Dimension getInterior(DrawingPanel panel) {
     panel.setPreferredMinMax(xmin,xmax,ymin,ymax);
     return dimension;
   }

  /**
   * Draws the image and the grid.
   * @param panel
   * @param g
   */
  public void draw(DrawingPanel panel, Graphics g) {
    g.drawImage(image, 1+panel.xToPix(xmin), 1+panel.yToPix(ymax),panel);
  }


}
