package org.opensourcephysics.display2d;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.*;
import javax.swing.event.TableModelEvent;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

public class GridDataTable extends JTable implements ActionListener {

   static final Color PANEL_BACKGROUND = javax.swing.UIManager.getColor("Panel.background");
   int refreshDelay = 0;                                                       // time in ms to delay refresh events
   javax.swing.Timer refreshTimer = new javax.swing.Timer(refreshDelay, this); // delay for refreshTable
   GridTableModel tableModel;
   RowNumberRenderer rowNumberRenderer = new RowNumberRenderer(this);
   public GridDataTable(GridData griddata, int component) {
      super();
      refreshTimer.setRepeats(false);
      refreshTimer.setCoalesce(true);
      tableModel = new GridTableModel(griddata, component);
      setModel(tableModel);
      setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
      String name = getColumnName(0);
      TableColumn column = getColumn(name);
      int width = 20;
      column.setMinWidth(width);
      column.setResizable(true);
      //column.setMaxWidth(width);
      column.setWidth(width);
      width = 60;
      for(int i = 1, n = getColumnCount(); i<n; i++) {
         name = getColumnName(i);
         column = getColumn(name);
         column.setMinWidth(width);
         column.setWidth(width);
         column.setResizable(true);
      }
      sizeColumnsToFit(0);
   }

   /**
    *  Performs the action for the refresh timer by refreshing the data in the DataTable.
    *
    * @param  evt
    */
   public void actionPerformed(ActionEvent evt) {
      tableChanged(new TableModelEvent(tableModel, TableModelEvent.HEADER_ROW));
   }

   /**
    *  Refresh the data in the DataTable, as well as other changes to the table,
    *  such as row number visibility. Changes to the TableModels displayed in the
    *  table will not be visible until this method is called.
    */
   public void refreshTable() {
      if(refreshDelay>0) {
         refreshTimer.start();
      } else {
         Runnable doRefreshTable = new Runnable() {

            public synchronized void run() {
               tableChanged(new TableModelEvent(tableModel, TableModelEvent.HEADER_ROW));
            }
         };
         if(SwingUtilities.isEventDispatchThread()) {
            doRefreshTable.run();
         } else {
            SwingUtilities.invokeLater(doRefreshTable);
         }
      }
   }

   /**
    *  Returns an appropriate renderer for the cell specified by this row and
    *  column. If the <code>TableColumn</code> for this column has a non-null
    *  renderer, returns that. If the <code>TableColumn</code> for this column has
    *  the same name as a name specified in the setMaximumFractionDigits method,
    *  returns the appropriate renderer. If not, finds the class of the data in
    *  this column (using <code>getColumnClass</code>) and returns the default
    *  renderer for this type of data.
    *
    * @param  row     Description of Parameter
    * @param  column  Description of Parameter
    * @return         The cellRenderer value
    */
   public TableCellRenderer getCellRenderer(int row, int column) {
      int i = convertColumnIndexToModel(column);
      if(i==0) {
         return rowNumberRenderer;
      }
      return getDefaultRenderer(getColumnClass(column));
   }

   private static class RowNumberRenderer extends JLabel implements TableCellRenderer {

      JTable table;

      /**
       *  RowNumberRenderer constructor
       *
       * @param  _table  Description of Parameter
       */
      public RowNumberRenderer(JTable _table) {
         super();
         table = _table;
         setHorizontalAlignment(SwingConstants.RIGHT);
         setOpaque(true); //make background visible.
         setForeground(Color.BLACK);
         setBackground(PANEL_BACKGROUND);
      }

      /**
       *  returns a JLabel that is highlighted if the row is selected.
       *
       * @param  table
       * @param  value
       * @param  isSelected
       * @param  hasFocus
       * @param  row
       * @param  column
       * @return
       */
      public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
          int row, int column) {
         if(column==0) {
            setBackground(PANEL_BACKGROUND);
         }
         setText(value.toString());
         return this;
      }
   }
}
