/*
 * The org.opensourcephysics.display package contains components for rendering
 * two-dimensional scalar and vector fields.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.ComponentSampleModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import javax.swing.JFrame;
import org.opensourcephysics.controls.XML;
import org.opensourcephysics.controls.XMLControl;
import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.Grid;
import org.opensourcephysics.display.InteractivePanel;
import org.opensourcephysics.display.MeasuredImage;
import org.opensourcephysics.display.axes.XAxis;
import org.opensourcephysics.display.axes.XYAxis;

/**
 * GrayscalePlot renders 2d data as a grayscale image.
 *
 * A grayscale plot looks similar to a grid plot with a grayscale color palette.
 * However, it uses a different rendering model.
 *
 * @author     Wolfgang Christian
 * @created    February 2, 2003
 * @version    1.0
 */
public class GrayscalePlot extends MeasuredImage implements Plot2D {

   GridData griddata;
   double floor, ceil;
   boolean autoscaleZ = true;
   short[] bwData;           // 16 bits grayscale
   Grid grid;
   private int ampIndex = 0; // amplitude index
   private JFrame legendFrame;

   /**
    * Constructs a checker field with the given width and height.
    */
   public GrayscalePlot(GridData griddata) {
      setGridData(griddata);
      update();
   }

   /**
    * Gets the x coordinate for the given index.
    *
    * @param i int
    * @return double the x coordinate
    */
   public double indexToX(int i) {
      return griddata.indexToX(i);
   }

   /**
    * Gets the y coordinate for the given index.
    *
    * @param i int
    * @return double the y coordinate
    */
   public double indexToY(int i) {
      return griddata.indexToY(i);
   }

   /**
    * Gets closest index from the given x  world coordinate.
    *
    * @param x double the coordinate
    * @return int the index
    */
   public int xToIndex(double x) {
      return griddata.xToIndex(x);
   }

   /**
    * Gets closest index from the given y  world coordinate.
    *
    * @param x double the coordinate
    * @return int the index
    */
   public int yToIndex(double y) {
      return griddata.yToIndex(y);
   }

   /**
    * Sets the data to new values.
    *
    * The grid is resized to fit the new data if needed.
    *
    * @param val
    */
   public void setAll(Object obj) {
      double[][] val = (double[][]) obj;
      copyData(val);
      update();
   }

   /**
    * Sets the values and the scale.
    *
    * The grid is resized to fit the new data if needed.
    *
    * @param val array of new values
    * @param xmin double
    * @param xmax double
    * @param ymin double
    * @param ymax double
    */
   public void setAll(Object obj, double xmin, double xmax, double ymin, double ymax) {
      double[][] val = (double[][]) obj;
      copyData(val);
      if(griddata.isCellData()) {
         griddata.setCellScale(xmin, xmax, ymin, ymax);
      } else {
         griddata.setScale(xmin, xmax, ymin, ymax);
      }
      setMinMax(xmin, xmax, ymin, ymax);
      update();
   }

   private void copyData(double val[][]) {
      if((griddata!=null)&&!(griddata instanceof ArrayData)) {
         throw new IllegalStateException("SetAll only supports ArrayData for data storage.");
      }
      if((griddata==null)||(griddata.getNx()!=val.length)||(griddata.getNy()!=val[0].length)) {
         griddata = new ArrayData(val.length, val[0].length, 1);
         setGridData(griddata);
      }
      double[][] data = griddata.getData()[0];
      int ny = data[0].length;
      for(int i = 0, nx = data.length; i<nx; i++) {
         System.arraycopy(val[i], 0, data[i], 0, ny);
      }
   }

   /**
    * Gets the GridData object.
    * @return GridData
    */
   public GridData getGridData() {
      return griddata;
   }

   /**
    * Sets the data storage to the given value.
    *
    * @param _griddata new data storage
    */
   public void setGridData(GridData _griddata) {
      griddata = _griddata;
      if(griddata==null) {
         return;
      }
      double[][][] data = griddata.getData();
      int nx = griddata.getNx();
      int ny = griddata.getNy();
      int size = nx*ny;
      Grid newgrid = new Grid(nx, ny, xmin, xmax, ymin, ymax);
      if(grid!=null) {
         newgrid.setColor(grid.getColor());
         newgrid.setVisible(grid.isVisible());
      } else {
         newgrid.setColor(Color.pink);
      }
      grid = newgrid;
      ComponentColorModel ccm = new ComponentColorModel(ColorSpace.getInstance(ColorSpace.CS_GRAY), new int[]{16},
         false, // hasAlpha
         false, // alspha premultiplied
         Transparency.OPAQUE, DataBuffer.TYPE_USHORT);
      ComponentSampleModel csm = new ComponentSampleModel(DataBuffer.TYPE_USHORT, nx, ny, 1, nx, new int[]{0});
      bwData = new short[size];
      DataBuffer databuffer = new DataBufferUShort((short[]) bwData, size);
      WritableRaster raster = Raster.createWritableRaster(csm, databuffer, new Point(0, 0));
      image = new BufferedImage(ccm, raster, true, null);
      xmin = griddata.getLeft();
      xmax = griddata.getRight();
      ymin = griddata.getBottom();
      ymax = griddata.getTop();
   }

   public JFrame showLegend() {
      InteractivePanel dp = new InteractivePanel();
      dp.setPreferredSize(new java.awt.Dimension(300, 66));
      dp.setGutters(0, 0, 0, 35);
      dp.setClipAtGutter(false);
      if(legendFrame==null) {
         legendFrame = new JFrame("Legend");
      }
      legendFrame.setResizable(false);
      legendFrame.setContentPane(dp);
      int numColors = 256;
      GridPointData pointdata = new GridPointData(numColors, 1, 1);
      double[][][] data = pointdata.getData();
      double delta = (ceil-floor)/(numColors);
      double cval = floor-delta/2;
      for(int i = 0, n = data.length; i<n; i++) {
         data[i][0][2] = cval;
         cval += delta;
      }
      pointdata.setScale(floor-delta, ceil+delta, 0, 1);
      GrayscalePlot cb = new GrayscalePlot(pointdata);
      cb.setShowGridLines(false);
      cb.setAutoscaleZ(false, floor, ceil);
      cb.update();
      dp.addDrawable(cb);
      XAxis xaxis = new XAxis("");
      xaxis.setLocationType(XYAxis.DRAW_AT_LOCATION);
      xaxis.setLocation(-0.5);
      xaxis.setEnabled(true);
      dp.addDrawable(xaxis);
      legendFrame.pack();
      legendFrame.setVisible(true);
      return legendFrame;
   }

   /**
    * Sets the autoscale flag and the floor and ceiling values for the colors.
    *
    * If autoscaling is true, then the min and max values of z are span the colors.
    *
    * If autoscaling is false, then floor and ceiling values limit the colors.
    * Values below min map to the first color; values above max map to the last color.
    *
    * @param isAutoscale
    * @param floor
    * @param ceil
    */
   public void setAutoscaleZ(boolean isAutoscale, double _floor, double _ceil) {
      autoscaleZ = isAutoscale;
      if(autoscaleZ) {
         update();
      } else {
         floor = _floor;
         ceil = _ceil;
      }
   }

   /**
    * Gets the autoscale flag for z.
    *
    * @return boolean
    */
   public boolean isAutoscaleZ() {
      return autoscaleZ;
   }

   /**
    * Gets the floor for scaling the z data.
    * @return double
    */
   public double getFloor() {
      return floor;
   }

   /**
    * Gets the ceiling for scaling the z data.
    * @return double
    */
   public double getCeiling() {
      return ceil;
   }

   /**
    * Sets the show grid option.
    *
    * @param  show
    */
   public void setShowGridLines(boolean showGrid) {
      grid.setVisible(showGrid);
   }

   /**
    * Updates the buffered image using the data array.
    */
   public void update() {
      if(griddata==null) {
         return;
      }
      if(autoscaleZ) {
         double[] minmax = griddata.getZRange(ampIndex);
         floor = minmax[0];
         ceil = minmax[1];
      }
      recolorImage();
   }

   /**
    * Sets the indexes for the data component that will be plotted.
    *
    * @param n the sample-component
    */
   public void setIndexes(int[] indexes) {
      ampIndex = indexes[0];
   }

   /**
    * Recolors the image pixels using the data array.
    */
   protected void recolorImage() {
      if(griddata==null) {
         return;
      }
      if(griddata.isCellData()) {
         double dx = griddata.getDx();
         double dy = griddata.getDy();
         xmin = griddata.getLeft()-dx/2;
         xmax = griddata.getRight()+dx/2;
         ymin = griddata.getBottom()+dy/2;
         ymax = griddata.getTop()-dy/2;
      } else {
         xmin = griddata.getLeft();
         xmax = griddata.getRight();
         ymin = griddata.getBottom();
         ymax = griddata.getTop();
      }
      grid.setMinMax(xmin, xmax, ymin, ymax);
      double[][][] data = griddata.getData();
      int nx = griddata.getNx();
      int ny = griddata.getNy();
      double zscale = 2*(int) Short.MAX_VALUE/(ceil-floor);
      if(griddata instanceof GridPointData) {
         int index = ampIndex+2;
         for(int ix = 0; ix<nx; ix++) {
            for(int iy = 0; iy<ny; iy++) {
               double val = zscale*(data[ix][iy][index]-floor);
               if(val<0) {
                  bwData[iy*nx+ix] = 0;
               } else if(val>2*(int) Short.MAX_VALUE) {
                  bwData[iy*nx+ix] = (short) (2*(int) Short.MAX_VALUE);
               } else {
                  bwData[iy*nx+ix] = (short) val;
               }
            }
         }
      } else if(griddata instanceof ArrayData) {
         for(int ix = 0; ix<nx; ix++) {
            for(int iy = 0; iy<ny; iy++) {
               double val = zscale*(data[ampIndex][ix][iy]-floor);
               if(val<0) {
                  bwData[iy*nx+ix] = 0;
               } else if(val>2*(int) Short.MAX_VALUE) {
                  bwData[iy*nx+ix] = (short) (2*(int) Short.MAX_VALUE);
               } else {
                  bwData[iy*nx+ix] = (short) val;
               }
            }
         }
      }
   }

   /**
    * Draws the image and the grid.
    * @param panel
    * @param g
    */
   public void draw(DrawingPanel panel, Graphics g) {
      if(griddata==null) {
         return;
      }
      super.draw(panel, g); // draws the image
      grid.draw(panel, g);
   }

   /**
    * Floor and ceiling colors are not supported.
    * Floor is black; ceiling is white.
    *
    * @param colors
    */
   public void setFloorCeilColor(Color floorColor, Color ceilColor) {}

   /**
    * Setting the color palette is not supported.  Palette is gray scale.
    * @param colors
    */
   public void setColorPalette(Color[] colors) {}

   /**
    * Setting the color palette is not supported.  Palette is gray scale.
    * @param colors
    */
   public void setPaletteType(int type) {}

   /**
    * Sets the grid color.
    *
    * @param c
    */
   public void setGridLineColor(Color c) {
      grid.setColor(c);
   }

   /**
    * Gets an XML.ObjectLoader to save and load data for this program.
    *
    * @return the object loader
    */
   public static XML.ObjectLoader getLoader() {
      return new Plot2DLoader() {

         public Object createObject(XMLControl control) {
            return new GrayscalePlot(null);
         }
      };
   }
}
