/*
 * The org.opensourcephysics.display package contains components for rendering
 * two-dimensional scalar and vector fields.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;

import java.awt.Graphics;
import java.util.ArrayList;
import java.util.Iterator;


 /**
 * Title:        Contour
 * Description:  The class <code>LineAccumulator</code> accumulates line drawing information and
 * then draws all accumulated lines together. It is used as contour accumulator
 * in the contour plot.
 *
 * The contour plot uses some code from the Surface Plotter package by Yanto Suryono.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */

public final class ContourAccumulator {
  private ArrayList accumulator;

  /**
   * The constructor of <code>LineAccumulator</code>
   */

  ContourAccumulator() {
    accumulator = new ArrayList();
  }

  /**
   * Adds a line to the accumulator.
   *
   * @param x1 the first point's x coordinate
   * @param y1 the first point's y coordinate
   * @param x2 the second point's x coordinate
   * @param y2 the second point's y coordinate
   */

  public synchronized void addLine(int x1, int y1, int x2, int y2) {
    accumulator.add(new LineRecord(x1,y1,x2,y2));
  }

  /**
   * Clears accumulator.
   */

  public synchronized void clearAccumulator() {
    accumulator.clear();
  }

  /**
   * Draws all accumulated lines.
   *
   * @param g the graphics context to draw
   */

  public void drawAll(Graphics g) {
    ArrayList tempList =null;
    synchronized(this){
      tempList = (ArrayList) accumulator.clone();
    }
    Iterator it = tempList.iterator();
    while (it.hasNext()) {
      LineRecord line = (LineRecord)it.next();
      g.drawLine(line.x1,line.y1,line.x2,line.y2);
    }

  }
}

/**
 * Represents a stright line.
 * Used by <code>LineAccumulator</code> class.
 *
 * @see LineAccumulator
 */

class LineRecord extends Object {
  /**
   * @param x1 the first point's x coordinate
   */
  public int x1;

  /**
   * @param y1 the first point's y coordinate
   */
  public int y1;

  /**
   * @param x2 the second point's x coordinate
   */
  public int x2;

  /**
   * @param y2 the second point's y coordinate
   */
  public int y2;

  /**
   * The constructor of <code>LineRecord</code>
   *
   * @param x1 the first point's x coordinate
   * @param y1 the first point's y coordinate
   * @param x2 the second point's x coordinate
   * @param y2 the second point's y coordinate
   */

  LineRecord(int x1, int y1, int x2, int y2) {
    super();
    this.x1 = x1; this.y1 = y1;
    this.x2 = x2; this.y2 = y2;
  }
}

