/*
* The org.opensourcephysics.display package contains components for rendering
* two-dimensional scalar and vector fields.
* Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;

import java.awt.Color;
import java.awt.Graphics;

import javax.swing.JFrame;

import org.opensourcephysics.display.DrawingPanel;

/**
 * ComplexInterpolatedPlot creates an image of a scalar field by inerpolating every
 * image pixel to an untabulated point (x,y) in the 2d data.  This interpolation smooths
 * the resulting image.
 *
 * @author     Wolfgang Christian
 * @created    February 2, 2003
 * @version    1.0
 */
public class ComplexContourPlot extends ComplexInterpolatedPlot{

  ContourPlot contour;
  boolean showContours=true;

  /**
   * Constructs the ComplexContourPlot using the given 2d datset.
   *
   * @param griddata GridData
   */
  public ComplexContourPlot(){
     super(null);
  }

  /**
   * Constructs the ComplexContourPlot using the given 2d datset.
   *
   * @param griddata GridData
   */
  public ComplexContourPlot(GridData griddata) {
    super(griddata);
    contour = new ContourPlot(griddata);
    contour.setPaletteType(ColorMapper.WIREFRAME);
    contour.setGridLineColor(Color.lightGray);
    contour.update();
  }

  /**
   * Sets the autoscale flag and the floor and ceiling values for the colors.
   *
   * If autoscaling is true, then the min and max values of z are span the colors.
   *
   * If autoscaling is false, then floor and ceiling values limit the colors.
   * Values below min map to the first color; values above max map to the last color.
   *
   * @param isAutoscale
   * @param floor
   * @param ceil
   */
  public void setAutoscaleZ(boolean isAutoscale, double floor, double ceil){
    super.setAutoscaleZ(isAutoscale, ceil);
    contour.setAutoscaleZ(isAutoscale,  floor, ceil);
  }

  /**
   * Updates the buffered image using the data array.
   */
  public void update (){
    super.update();
    if(contour!=null && showContours) contour.update();
  }

  /**
   * Sets the data storage to the given value.
   *
   * @param griddata
   */
  public void setGridData(GridData griddata) {
    super.setGridData(griddata);
    super.setShowGridLines(false);
    contour.setGridData(griddata);
  }

  /**
   * Sets the indexes for the data component that will be plotted.
   *
   * @param n the sample-component
   */
  public void setIndexes(int[] indexes) {
    super.setIndexes(indexes);
    contour.setIndexes(indexes);
  }


  /**
   * Sets the visibility of the contour plot.
   * Drawing will be disabled if visible is false.
   *
   * @param isVisible
   */
  public void setVisible(boolean isVisible){
    visible=isVisible;
  }

  /**
   * Shows how values map to colors.
   */
  public JFrame showLegend(){
    return colorMap.showLegend();
  }


  /**
   * Shows the contour lines.
   *
   * @param show
   */
  public void setShowGridLines(boolean show){
    contour.setShowGridLines(show);
  }

  /**
   * Setting the color palette is not supported.  The complex palette always maps phase to color.
   * @param colors
   */
  public void setColorPalette(Color[] colors) {
  }

  /**
   * Setting the palette is not supported.   The complex palette always maps phase to color.
   * @param type
   */
  public void setPaletteType(int type) {
  }


  /**
   * Sets the floor and ceiling colors.
   *
   * @param floorColor
   * @param ceilColor
   */
  public void setFloorCeilColor (Color floorColor, Color ceilColor){
    super.setFloorCeilColor(floorColor, ceilColor);
  }

  /**
   * Sets the contour line color.
   * The default line color is dark green.
   * @param color
   */
  public void setGridLineColor(Color color){
    contour.setGridLineColor( color);
  }


  /**
   * Draws the image and the grid.
   * @param panel
   * @param g
   */
  public void draw (DrawingPanel panel, Graphics g) {
    if(!visible) return;
    super.draw(panel,g);
    if(showContours) contour.draw(panel, g);
  }

}
