/*
* The org.opensourcephysics.display package contains components for rendering
* two-dimensional scalar and vector fields.
* Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display2d;
import java.awt.Color;

import javax.swing.JFrame;

import org.opensourcephysics.display.InteractivePanel;
import org.opensourcephysics.display.axes.XAxis;
import org.opensourcephysics.display.axes.XYAxis;

public class ComplexColorMapper{
  static final double PI2=Math.PI*2;
  static final double COLOR_ERR=100*Float.MIN_VALUE;
  private double ceil;
  private Color ceilColor=Color.lightGray;
  private JFrame legendFrame;
  int[]    reds   = new int[256];
  int[]    greens = new int[256];
  int[]    blues  = new int[256];


  public  ComplexColorMapper (double _ceil) {
    ceil=_ceil;
    initColors();
  }

  public static JFrame showPhaseLegend() {
    InteractivePanel dp = new InteractivePanel();
    dp.setPreferredSize(new java.awt.Dimension(300, 66));
    dp.setGutters(0, 0, 0, 35);
    dp.setClipAtGutter(false);
    JFrame legendFrame = new JFrame("Complex Phase");
    legendFrame.setResizable(false);
    legendFrame.setContentPane(dp);
    int numPts = 360;
    GridPointData pointdata = new GridPointData(numPts, 1, 3);
    double[][][] data = pointdata.getData();
    double theta = -Math.PI, delta = 2 * Math.PI / (numPts);
    for (int i = 0, n = data.length; i < n; i++) {
      data[i][0][2] = 0.999;
      data[i][0][3] = Math.cos(theta);
      data[i][0][4] = Math.sin(theta);
      theta += delta;
    }
    pointdata.setScale( -Math.PI, Math.PI, 0, 1);
    Plot2D plot = new ComplexGridPlot(pointdata);
    plot.setShowGridLines(false);
    plot.update();
    dp.addDrawable(plot);
    XAxis xaxis = new XAxis("");
    xaxis.setLocationType(XYAxis.DRAW_AT_LOCATION);
    xaxis.setLocation( -0.5);
    xaxis.setEnabled(true);
    dp.addDrawable(xaxis);
    legendFrame.pack();
    legendFrame.setVisible(true);
    return legendFrame;
  }

  /**
   * Shows the phase legend.
   */
  public JFrame showLegend() {
    InteractivePanel dp = new InteractivePanel();
    dp.setPreferredSize(new java.awt.Dimension(300, 66));
    dp.setGutters(0,0,0,35);
    dp.setClipAtGutter(false);
    if(legendFrame==null) legendFrame=new JFrame("Complex Phase");
    legendFrame.setResizable(false);
    legendFrame.setContentPane(dp);
    int numPts=360;
    GridPointData pointdata = new GridPointData(numPts, 1, 3);
    double[][][] data = pointdata.getData();
    double theta=-Math.PI, delta = 2*Math.PI /(numPts);
    for (int i = 0, n = data.length; i < n; i++) {
      data[i][0][2] = 0.999;
      data[i][0][3] = Math.cos(theta);
      data[i][0][4] = Math.sin(theta);
      theta += delta;
    }
    pointdata.setScale(-Math.PI,Math.PI,0,1);
    Plot2D plot = new ComplexGridPlot(pointdata);
    plot.setShowGridLines(false);
    plot.update();
    dp.addDrawable(plot);
    XAxis xaxis = new XAxis("");
    xaxis.setLocationType(XYAxis.DRAW_AT_LOCATION);
    xaxis.setLocation(-0.5);
    xaxis.setEnabled(true);
    dp.addDrawable(xaxis);
    legendFrame.pack();
    legendFrame.setVisible(true);
    return legendFrame;
  }


  /**
   * Sets the scale.
   * @param floor
   */
  public void setScale (double _ceil){
    ceil=_ceil;
  }

  /**
   * Converts a double to color components.
   * @param mag
   * @return the color
   */
  public byte[] samplesToComponents (double[] samples, byte[] rgb) {
    Color color=samplesToColor (samples);
    rgb[0]=(byte)color.getRed();
    rgb[1]=(byte)color.getGreen();
    rgb[2]=(byte)color.getBlue();
    return rgb;
  }

  /**
   * Converts a phase angle in the range [-Pi,Pi] to hue, saturation, and brightness.
   *
   * @param phi phase angle
   * @return the HSB color
   */
  public Color phaseToColor(double phi) {
    float b = 1; // brightness
    float h = (float) ( (Math.PI + phi) / PI2);
    int index = ( (int) (255 * h));
    return new Color( (int) (b * reds[index]), (int) (b * greens[index]),
                     (int) (b * blues[index]));
  }


  /**
   * Converts a complex number to hue, saturation, and brightness.
   * @param re
   * @param im
   * @return the HSB color
   */
  public Color complexToColor(double re, double im) {
    float b = 1; // brightness
    float h = (float) ( (Math.PI + Math.atan2(im, re)) / PI2);
    int index = ( (int) (255 * h));
    return new Color( (int) (b * reds[index]), (int) (b * greens[index]),
                     (int) (b * blues[index]));
  }


  /**
   * Converts an array of samples to hue, saturation, and brightness.
   * Samples contains magnitude, re, and im.
   * @param samples
   * @return the HSB color
   */
  public Color samplesToColor (double[] samples) {
    if (samples[0] <=0)
      return  Color.black;
    else if ( samples[0]>ceil+COLOR_ERR) {
      return  ceilColor;
    }
    float b = (float)(samples[0]/ceil); // brightness
    float h = (float)((Math.PI+Math.atan2(samples[2],samples[1]))/PI2); // hue
    int index=((int)(255*h));
    return  new Color((int)(b*reds[index]),(int)(b*greens[index]),(int)(b*blues[index]));
  }

  /**
   * Converts a vertex point array of samples to hue, saturation, and brightness.
   *
   * A vertex containing x, y, magnitude, re, and im.
   *
   * @param vertex the point
   * @return the HSB color
   */
  public Color pointToColor (double[] vertex) {
    if (vertex[2] <=0)
      return  Color.black;
    else if ( vertex[2]>ceil+COLOR_ERR) {
      return  ceilColor;
    }
    float b = (float)(vertex[2]/ceil);
    float h = (float)((Math.PI+Math.atan2(vertex[4],vertex[3]))/PI2);
    int index=((int)(255*h));
    return  new Color((int)(b*reds[index]),(int)(b*greens[index]),(int)(b*blues[index]));
    //return  Color.getHSBColor(h,1,b);
  }

  /**
   * Gets the ceiling color.
   * @return
   */
  public double getCeil() {
    return ceil;
}


  /**
   * Gets the ceiling color.
   * @return
   */
  public Color getCeilColor(){
    return ceilColor;
  }


  /**
   * Sets the ceiling.
   *
   * @param floorColor
   * @param ceilColor
   */
  public void setCeilColor (Color _ceilColor){
    ceilColor=_ceilColor;
  }

  private void initColors(){
      double pi=Math.PI;
      for(int i=0; i<256; i++){
          double val=Math.abs(Math.sin(pi*i/255));
          blues[i]=(int)(255*val*val);
          val=Math.abs(Math.sin(pi*i/255+pi/3));
          greens[i]=(int)(255*val*val*Math.sqrt(val));
          val=Math.abs(Math.sin(pi*i/255+2*pi/3));
          reds[i]=(int)(255*val*val);
      }
  }


}
