/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package  org.opensourcephysics.display.axes;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;

import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.Interactive;

/**
 * YAxis: a y axis that knows how to draw itself in a drawing panel.
 * Copyright:    Copyright (c) 2005  Gould, Christian, and Tobochnik
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class YAxis extends XYAxis{
  AffineTransform rot90= AffineTransform.getRotateInstance(-Math.PI/2);

  public YAxis(){
    this("Y Axis");
  }

  public YAxis(String title){
    super();
    setTitle(title);
    axisLabel.setTheta(Math.PI/2);
  }

  /**
   * Draws the axis in the drawing panel.
   * @param drawingPanel
   * @param g
   */
  public void draw(DrawingPanel drawingPanel, Graphics g) {
    int pixLoc=drawingPanel.xToPix(location);
    if (pixLoc < 1) {
      location = drawingPanel.getXMin();
    }
    if (pixLoc > drawingPanel.getWidth() - 1) {
      location = drawingPanel.getXMax();
    }
    Graphics2D g2 = (Graphics2D) g;
    Shape clipShape = g2.getClip();
    g2.clipRect(0, 0, drawingPanel.getWidth(), drawingPanel.getHeight());

    switch (locationType) {
      case DRAW_AT_LOCATION:
      case DRAW_IN_DISPLAY:
        drawInsideDisplay(drawingPanel, g);
        break;
      case DRAW_IN_GUTTER:
        drawInsideGutter(drawingPanel, g);
        break;
      default:
        drawInsideDisplay(drawingPanel, g);
        break;
    }
    g2.setClip(clipShape);
  }

  private void drawInsideDisplay(DrawingPanel drawingPanel, Graphics g) {
    int bottomGutter=drawingPanel.getBottomGutter();
    int rightGutter=drawingPanel.getRightGutter();
    int leftGutter=drawingPanel.getLeftGutter();
    int topGutter=drawingPanel.getTopGutter();
    FontMetrics fm=g.getFontMetrics();
    int sw=0; // the string width
    g.setColor(color);
    if(locationType==XYAxis.DRAW_IN_DISPLAY)
      location=(drawingPanel.getXMax()+drawingPanel.getXMin())/2;
    int xo=drawingPanel.xToPix(location);
    int yo=drawingPanel.getHeight()-bottomGutter;
    int w=drawingPanel.getWidth()-leftGutter-rightGutter;
    int h=drawingPanel.getHeight()-bottomGutter-topGutter;
    calculateLabels(drawingPanel.getYMin(),drawingPanel.getYMax(),1+h/35);
    String[] temp_strings=label_string;  // get a reference for thread safety
    double[] temp_values=label_value;    // get a reference for thread safety
    if(temp_strings.length!=temp_values.length) return;
    for(int i=0, n=temp_values.length; i<n; i++) {
      if(axisType==LINEAR){
        int ypix=drawingPanel.yToPix(temp_values[i]*decade_multiplier);
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(leftGutter, ypix,drawingPanel.getWidth()- rightGutter-2 , ypix);
          g.setColor(color);
        }
        g.drawLine(xo-5, ypix, xo+5, ypix);
        sw=fm.stringWidth(temp_strings[i]);
        g.drawString(temp_strings[i], xo-sw-7, ypix+5);
      }
      else{  // log axis
        int ypix=drawingPanel.yToPix(Math.pow(10,temp_values[i]*decade_multiplier));
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(leftGutter, ypix,drawingPanel.getWidth()- rightGutter-2 , ypix);
          g.setColor(color);
        }
        g.drawLine(xo-5, ypix, xo+5, ypix);
        sw=fm.stringWidth(logBase);
        drawMultiplier(xo-sw-7, ypix+5, (int)temp_values[i] ,(Graphics2D) g);
      }
    }
    g.drawLine(xo, yo,xo, yo-h);
    Graphics2D g2= (Graphics2D)g;
    if(axisType==LINEAR && label_exponent!=0){
      g2.setColor(Color.red);
      g2.drawString("x10",5 ,18);
      g2.setFont(g2.getFont().deriveFont(Font.PLAIN,9.0f));
      g2.drawString(""+label_exponent,25 ,12);
    }
    g2.setColor(Color.black);
    if (axisLabel!=null){
       axisLabel.setY((drawingPanel.getYMax()+drawingPanel.getYMin())/2);
       //axisLabel.setX(drawingPanel.getXMin()-(leftGutter-20)/drawingPanel.getXPixPerUnit());
       axisLabel.setX(drawingPanel.pixToX(Math.max(leftGutter/2-10, 18)));
       axisLabel.draw(drawingPanel, g2);
    }
  }

  private void drawInsideGutter(DrawingPanel drawingPanel, Graphics g) {
    int bottomGutter=drawingPanel.getBottomGutter();
    int rightGutter=drawingPanel.getRightGutter();
    int leftGutter=drawingPanel.getLeftGutter();
    int topGutter=drawingPanel.getTopGutter();
    FontMetrics fm=g.getFontMetrics();
    int sw=0; // the string width
    g.setColor(color);
    int xo=leftGutter;
    int yo=drawingPanel.getHeight()-bottomGutter-1;
    int w=drawingPanel.getWidth()-leftGutter-rightGutter;
    int h=drawingPanel.getHeight()-bottomGutter-topGutter;
    g.drawLine(xo, yo,xo, yo-h);
    calculateLabels(drawingPanel.getYMin(),drawingPanel.getYMax(),1+h/35);
    String[] temp_strings=label_string;  // get a reference for thread safety
    double[] temp_values=label_value;    // get a reference for thread safety
    if(temp_strings.length!=temp_values.length) return;
    for(int i=0, n=temp_values.length; i<n; i++) {
      if(axisType==LINEAR){
        int ypix=drawingPanel.yToPix(temp_values[i]*decade_multiplier);
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xo, ypix,drawingPanel.getWidth()- rightGutter-2 , ypix);
          g.setColor(color);
        }
        g.drawLine(xo-5, ypix, xo, ypix);
        sw=fm.stringWidth(temp_strings[i]);
        g.drawString(temp_strings[i], xo-sw-7, ypix+5);
      }
      else{  // log axis
        int ypix=drawingPanel.yToPix(Math.pow(10,temp_values[i]*decade_multiplier));
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xo, ypix,drawingPanel.getWidth()- rightGutter-2 , ypix);
          g.setColor(color);
        }
        g.drawLine(xo-5, ypix, xo, ypix);
        sw=fm.stringWidth(logBase);
        drawMultiplier(xo-sw-14, ypix+5, (int)temp_values[i] ,(Graphics2D) g);
      }
    }
    Graphics2D g2= (Graphics2D)g;
    if(axisType==LINEAR && label_exponent!=0){
      g2.setColor(Color.red);
      g2.drawString("x10",5 ,18);
      g2.setFont(g2.getFont().deriveFont(Font.PLAIN,9.0f));
      g2.drawString(""+label_exponent,25 ,12);
    }
    g2.setColor(Color.black);
    if(axisLabel!=null){
       axisLabel.setY((drawingPanel.getYMax()+drawingPanel.getYMin())/2);
       //axisLabel.setX(drawingPanel.getXMin()-(leftGutter-20)/drawingPanel.getXPixPerUnit());
       axisLabel.setX(drawingPanel.pixToX(Math.max(leftGutter/2-10, 18)));
       axisLabel.draw(drawingPanel, g2);
    }
  }

  // implements interactive drawable interface
  public Interactive findInteractive(DrawingPanel panel, int xpix, int ypix) {
    if(!enabled) return null;
    if (Math.abs(panel.xToPix(location) - xpix) < 2)
      return this;
    return null;
  }

  public void setXY(double x, double y) {
    location = x;
  }

  public void setX(double x) {
    location = x;
}

}
