/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package  org.opensourcephysics.display.axes;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;

import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.Interactive;

/**
 * XAxis: an x axis that knows how to draw itself in a drawing panel.
 * Copyright:    Copyright (c) 2005  Gould, Christian, and Tobochnik
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class XAxis extends XYAxis{

  public XAxis(){
    this("X Axis");
  }

  public XAxis(String title){
    super();
    setTitle(title);
  }

  /**
   * Draws the axis in the drawing panel.
   * @param drawingPanel
   * @param g
   */

  public void draw(DrawingPanel drawingPanel, Graphics g) {
    int pixLoc=drawingPanel.yToPix(location);
    if(pixLoc<1){
      location = drawingPanel.getYMin();
    }
    if (pixLoc >drawingPanel.getHeight()-1) {
      location = drawingPanel.getYMax();
    }
     Graphics2D g2        = (Graphics2D) g;
     Shape      clipShape = g2.getClip();
     g2.clipRect(0, 0, drawingPanel.getWidth(), drawingPanel.getHeight());
     switch (locationType){
       case DRAW_AT_LOCATION:
       case DRAW_IN_DISPLAY:
        drawInsideDisplay( drawingPanel, g);
        break;
      case DRAW_IN_GUTTER:
        drawInsideGutter( drawingPanel, g);
        break;
      default:
        drawInsideDisplay( drawingPanel, g);
        break;
     }
     g2.setClip(clipShape);
  }

  private void drawInsideDisplay(DrawingPanel drawingPanel, Graphics g) {
    int bottomGutter=drawingPanel.getBottomGutter();
    int rightGutter=drawingPanel.getRightGutter();
    int leftGutter=drawingPanel.getLeftGutter();
    int topGutter=drawingPanel.getTopGutter();
    FontMetrics fm=g.getFontMetrics();
    int sw=0; // the string width
    g.setColor(color);
    if(locationType==XYAxis.DRAW_IN_DISPLAY)
      location=(drawingPanel.getYMax()+drawingPanel.getYMin())/2;
    int xo=leftGutter;
    int yo=drawingPanel.yToPix(location);
    int w=drawingPanel.getWidth()-leftGutter-rightGutter;
    int h=drawingPanel.getHeight()-bottomGutter-topGutter;
    g.drawLine(xo, yo, xo+w, yo);
    calculateLabels(drawingPanel.getXMin(),drawingPanel.getXMax(),1+w/35);
    String[] temp_strings=label_string;  // get a reference for thread safety
    double[] temp_values=label_value;    // get a reference for thread safety
    if(temp_strings.length!=temp_values.length) return;
    for(int i=0, n=temp_strings.length; i<n; i++) {
      if(axisType==LINEAR){
        int xpix=drawingPanel.xToPix(temp_values[i]*decade_multiplier);
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xpix, topGutter+1, xpix, drawingPanel.getHeight()- bottomGutter-1);
          g.setColor(color);
        }
        g.drawLine(xpix, yo-5, xpix, yo+5);
        sw=fm.stringWidth(temp_strings[i]);
        g.drawString(temp_strings[i], xpix-sw/2, yo +18);
      }
      else{  // log axis
        int xpix=drawingPanel.xToPix(Math.pow(10,temp_values[i]*decade_multiplier));
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xpix, topGutter+1, xpix, drawingPanel.getHeight()- bottomGutter-1);
          g.setColor(color);
        }
        g.drawLine(xpix, yo-5, xpix, yo+5);
        sw=fm.stringWidth(logBase);
        drawMultiplier(xpix-sw/2, yo +18, (int)temp_values[i] ,(Graphics2D) g);
      }
    }
    int ypix=drawingPanel.getHeight()-Math.max(bottomGutter/2,6);
    Graphics2D g2= (Graphics2D)g;
    Font oldFont=g2.getFont();
    if(axisType==LINEAR && label_exponent!=0){
      g2.setColor(Color.red);
      g2.drawString("x10",drawingPanel.getWidth() -36 ,ypix);
      g2.setFont(g2.getFont().deriveFont(Font.PLAIN,9.0f));
      g2.drawString(""+label_exponent,drawingPanel.getWidth() -16 ,ypix-6);
    }
    g2.setColor(Color.black);
    if(axisLabel!=null){
       axisLabel.setX((drawingPanel.getXMax()+drawingPanel.getXMin())/2);
       axisLabel.setY(drawingPanel.getYMin()-20/drawingPanel.getYPixPerUnit());
       axisLabel.draw(drawingPanel,g2);
    }
  }


  private void drawInsideGutter(DrawingPanel drawingPanel, Graphics g) {
    int bottomGutter=drawingPanel.getBottomGutter();
    int rightGutter=drawingPanel.getRightGutter();
    int leftGutter=drawingPanel.getLeftGutter();
    int topGutter=drawingPanel.getTopGutter();
    FontMetrics fm=g.getFontMetrics();
    int sw=0; // the string width
    g.setColor(color);
    int xo=leftGutter;
    int yo=drawingPanel.getHeight()-bottomGutter-1;
    int w=drawingPanel.getWidth()-leftGutter-rightGutter;
    calculateLabels(drawingPanel.getXMin(),drawingPanel.getXMax(),1+w/35);
    String[] temp_strings=label_string;  // get a reference for thread safety
    double[] temp_values=label_value;    // get a reference for thread safety
    if(temp_strings.length!=temp_values.length) return;
    for(int i=0, n=temp_strings.length; i<n; i++) {
      if(axisType==LINEAR){
        int xpix=drawingPanel.xToPix(temp_values[i]*decade_multiplier);
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xpix, topGutter+1, xpix, yo);
          g.setColor(color);
        }
        g.drawLine(xpix, yo, xpix, yo+5);
        sw=fm.stringWidth(temp_strings[i]);
        g.drawString(temp_strings[i], xpix-sw/2, yo +18);
      }
      else{  // log axis
        int xpix=drawingPanel.xToPix(Math.pow(10,temp_values[i]*decade_multiplier));
        if(showMajorGrid){
          g.setColor(majorGridColor);
          g.drawLine(xpix, topGutter+1, xpix, yo);
          g.setColor(color);
        }
        g.drawLine(xpix, yo, xpix, yo+5);
        sw=fm.stringWidth(logBase);
        drawMultiplier(xpix-sw/2, yo +18, (int)temp_values[i] ,(Graphics2D) g);
      }
    }
    g.drawLine(xo, yo, xo+w, yo);
    int ypix=drawingPanel.getHeight()-Math.max(bottomGutter/2-15,6);
    Graphics2D g2= (Graphics2D)g;
    if(axisType==LINEAR && label_exponent!=0){
      g2.setColor(Color.red);
      g2.drawString("x10",drawingPanel.getWidth() -36 ,ypix);
      g2.setFont(g2.getFont().deriveFont(Font.PLAIN,9.0f));
      g2.drawString(""+label_exponent,drawingPanel.getWidth() -16 ,ypix-6);
    }
    g2.setColor(Color.black);
    if(axisLabel!=null){
       axisLabel.setX((drawingPanel.getXMax()+drawingPanel.getXMin())/2);
       axisLabel.setY(drawingPanel.pixToY(drawingPanel.getHeight()-Math.max(bottomGutter/2-10, 10)));
       axisLabel.draw(drawingPanel, g2);
    }
  }

  // implements interactive interface
  public Interactive findInteractive(DrawingPanel panel, int xpix, int ypix) {
    if(!enabled) return null;
    if (Math.abs(panel.yToPix(location) - ypix) < 2)
      return this;
    return null;
  }

  public void setXY(double x, double y) {
    location = y;
  }

  public void setY(double y) {
    location = y;
}

}
