/*
 *  The org.opensourcephysics.display package contains the drawing framework
 *  for the book Simulations in Physics.  This framework defines objects that
 *  implement the Drawable interface and a DrawingPanel for rendering these objects.
 *  Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display.axes;

import java.awt.Color;
import java.awt.Graphics;
import java.util.ArrayList;
import java.util.Iterator;
import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.PlottingPanel;
import java.awt.Font;
import org.opensourcephysics.display.DrawableTextLine;

/**
 * CustomAxes defines a simple axes with an interior and a title.
 *
 * This object can be customized by adding drawable components.
 *
 * @author W. Christian
 */
public class CustomAxes implements DrawableAxes {

  Color        gridColor     = Color.lightGray;
  Color        interiorColor = Color.white;
  DrawableTextLine titleLine = new DrawableTextLine("", 0, 0);
  Font titleFont = new Font("Helvetica", Font.BOLD, 14);
  boolean      visible      = true;
  ArrayList    drawableList = new ArrayList();  // list of Drawable objects

  /**
   * Constructor CustomAxes
   *
   * @param panel
   */
  public CustomAxes(PlottingPanel panel) {
    if(panel == null) {
      return;
    }
    panel.setGutters(25, 25, 25, 25);
    titleLine.setJustification(DrawableTextLine.CENTER);
    titleLine.setFont(titleFont);
    panel.setCoordinateStringBuilder( CoordinateStringBuilder.createCartesian());
    panel.setAxes(this);
  }

  /**
   * Sets the x label of the axes.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */

  public void setXLabel(String s, String font_name) {}

  /**
   * Sets the y label of the axes.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */
  public void setYLabel(String s, String font_name) {}

  /**
 * Gets the x axis label.
 *
 * @return String
 */
public String getXLabel() {
  return "";
}

/**
 * Gets the y axis label.
 *
 * @return String
 */
public String getYLabel() {
  return "";
}

/**
 * Gets the title.
 *
 * @return String
 */
public String getTitle() {
  return titleLine.getText();
}


  /**
   * Set a title that will be drawn within the drawing panel.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */
  public void setTitle(String s, String font_name) {
     titleLine.setText(s);
     if ((font_name==null)||font_name.equals("")){
        return;
     }
     titleLine.setFont(Font.decode(font_name));
  }

  /**
   * Sets the x axis to linear or logarithmic.
   *
   * @param isLog true for log scale; false otherwise
   */
  public void setXLog(boolean isLog) {}

  /**
   * Sets the y axis to linear or logarithmic.
   *
   * @param isLog true for log scale; false otherwise
   */
  public void setYLog(boolean isLog) {}

  /**
   * Sets the visibility of the axes.
   *
   * @param isVisible true if the axes are visible
   */
  public void setVisible(boolean isVisible) {
    visible = isVisible;
  }

  /**
   * Sets the interior background color.
   * @param color
   */
  public void setInteriorBackground(Color color) {
    interiorColor = color;
  }

  /**
   * Shows a grid line for every x axis major tickmark.
   * @param showGrid
   */
  public void setShowMajorXGrid(boolean showGrid) {}

  /**
   * Shows a grid line for every x axis minor tickmark.
   * @param showGrid
   */
  public void setShowMinorXGrid(boolean showGrid) {}

  /**
   * Shows a grid line for every y axis major tickmark.
   * @param showGrid
   */
  public void setShowMajorYGrid(boolean showGrid) {}

  /**
   * Shows a grid line for every y axis minor tickmark.
   * @param showGrid
   */
  public void setShowMinorYGrid(boolean showGrid) {}

  /**
   * Adds a drawable object to the drawable list.
   * @param drawable
   */
  public synchronized void addDrawable(Drawable drawable) {
    if((drawable != null) &&!drawableList.contains(drawable)) {
      drawableList.add(drawable);
    }
  }

  /**
   * Draws the axes in a drawing panel.
   * @param panel
   * @param g
   */
  public void draw(DrawingPanel panel, Graphics g) {
    if(!visible) {
      return;
    }
    if(interiorColor != panel.getBackground()) {
      g.setColor(interiorColor);
      int gw = panel.getLeftGutter() + panel.getRightGutter();
      int gh = panel.getTopGutter() + panel.getLeftGutter();
      g.fillRect(panel.getLeftGutter(), panel.getTopGutter(), panel.getWidth() - gw, panel.getHeight() - gh);
      g.setColor(gridColor);
      g.drawRect(panel.getLeftGutter(), panel.getTopGutter(), panel.getWidth() - gw, panel.getHeight() - gh);
    }
    Iterator   it        = drawableList.iterator();
    while(it.hasNext()) {
      Drawable drawable = (Drawable) it.next();
      drawable.draw(panel, g);
    }
    titleLine.setX((panel.getXMax()+panel.getXMin())/2);
    if(panel.getTopGutter()>20)titleLine.setY( panel.getYMax()+5/panel.getYPixPerUnit());
    else titleLine.setY( panel.getYMax()-25/panel.getYPixPerUnit());
    titleLine.draw(panel,g);
  }
}
