/*
 *  The org.opensourcephysics.display package contains the drawing framework
 *  for the book Simulations in Physics.  This framework defines objects that
 *  implement the Drawable interface and a DrawingPanel for rendering these objects.
 *  Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package  org.opensourcephysics.display.axes;

import java.awt.Color;
import java.awt.Graphics;
import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.PlottingPanel;
import org.opensourcephysics.display.DrawableTextLine;
import java.awt.Font;

/**
 * XYAxes draws X and Y axes in a plotting panel.
 *
 * @author W. Christian
 */
public class CartesianType2 implements CartesianAxes {

  XAxis        xaxis;
  YAxis        yaxis;
  DrawableTextLine titleLine = new DrawableTextLine("", 0, 0);
  Font titleFont = new Font("Helvetica", Font.BOLD, 14);
  boolean      visible = true;
  Color        interiorColor=Color.white;
  boolean      xlog=false, ylog=false; // Added by Paco


  /**
   * CartesianType2 draws axes in the drawing panel's gutter.
   *
   * The default gutters are set to 50, 25, 25, 5.
   *
   * @param  panel  the drawing panel that will use the axes
   * @param  hasGutter  true if axes will be drawn inside gutter; false otherwise
   */
  public CartesianType2(PlottingPanel panel) {
    titleLine.setJustification(DrawableTextLine.CENTER);
    titleLine.setFont(titleFont);
    xaxis = new XAxis();
    yaxis = new YAxis();
    xaxis.setLocationType(XYAxis.DRAW_IN_GUTTER);
    yaxis.setLocationType(XYAxis.DRAW_IN_GUTTER);
    panel.setGutters(50, 25, 25, 50);
    if(panel==null) return;
    panel.setCoordinateStringBuilder( CoordinateStringBuilder.createCartesian());
    panel.setAxes(this);
  }

  /**
   * Sets the visibility of the axes.
   *
   * @param isVisible true if the axes are visible
   */
  public void setVisible(boolean isVisible) {
    visible = isVisible;
  }

  /**
   * Sets the x label of the axes.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */
  public void setXLabel(String s, String font_name) {
    xaxis.setTitle(s,font_name);
  }

  /**
   * Sets the y label of the axes.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */
  public void setYLabel(String s, String font_name) {
    yaxis.setTitle(s, font_name);
  }

  /**
   * Gets the x axis label.
   *
   * @return String
   */
  public String getXLabel() {
    return xaxis.axisLabel.getText();
  }

  /**
   * Gets the y axis label.
   *
   * @return String
   */
  public String getYLabel() {
    return yaxis.axisLabel.getText();
  }

  /**
   * Gets the title.
   *
   * @return String
   */
  public String getTitle() {
    return titleLine.getText();
  }


  /**
   * Set the title.
   *
   * The title is drawn centered near the top of the drawing panel.
   * The font names understood are those understood by java.awt.Font.decode().
   * If the font name is null, the font remains unchanged.
   *
   * @param  label the label
   * @param font_name an optional font name
   */
  public void setTitle(String s, String font_name){
     titleLine.setText(s);
     if ((font_name==null)||font_name.equals("")){
        return;
     }
     titleLine.setFont(Font.decode(font_name));
  }

  /**
   * Sets the x axis to linear or logarithmic.
   *
   * @param isLog true for log scale; false otherwise
   */
  public void setXLog(boolean isLog) {
    xlog = isLog; // Added by Paco
    if(isLog) {
      xaxis.setAxisType(XYAxis.LOG10);
    } else {
      xaxis.setAxisType(XYAxis.LINEAR);
    }
  }

  /**
   * Sets the y axis to linear or logarithmic.
   *
   * @param isLog true for log scale; false otherwise
   */
  public void setYLog(boolean isLog) {
    ylog = isLog; // Added by Paco
    if(isLog) {
      yaxis.setAxisType(XYAxis.LOG10);
    } else {
      yaxis.setAxisType(XYAxis.LINEAR);
    }
  }

  public boolean isXLog() { return xlog; } // Added by Paco

  public boolean isYLog() { return ylog; } // Added by Paco


  /**
   * Draws the axes in the drawing panel.
   *
   * @param panel
   * @param g
   */
  public void draw(DrawingPanel panel, Graphics g) {
    if(!visible) {
      return;
    }
    if(interiorColor!=panel.getBackground()){
      g.setColor(interiorColor);
      int gw=panel.getLeftGutter()+panel.getRightGutter();
      int gh=panel.getTopGutter()+panel.getBottomGutter();
      g.fillRect(
          panel.getLeftGutter(),
          panel.getTopGutter(),
          panel.getWidth()-gw,
          panel.getHeight()-gh
          );
      g.setColor(Color.lightGray);
      g.drawRect(
          panel.getLeftGutter(),
          panel.getTopGutter(),
          panel.getWidth()-gw-1,
          panel.getHeight()-gh-1
          );
    }
    xaxis.draw(panel, g);
    yaxis.draw(panel, g);
    titleLine.setX((panel.getXMax()+panel.getXMin())/2);
    if(panel.getTopGutter()>20)titleLine.setY( panel.getYMax()+5/panel.getYPixPerUnit());
    else titleLine.setY( panel.getYMax()-25/panel.getYPixPerUnit());
    titleLine.draw(panel,g);
  }

  /**
   * Sets the interior background color.
   */
  public void setInteriorBackground(Color color){
    interiorColor=color;
  }

  /**
   * Shows a grid line for every x axis major tickmark.
   */
  public void setShowMajorXGrid(boolean showGrid){
     xaxis.setShowMajorGrid(showGrid);
     if(!showGrid) setShowMinorXGrid(showGrid);
  }

  /**
   * Shows a grid line for every x axis minor tickmark.
   */
  public void setShowMinorXGrid(boolean showGrid){
    // minor grids not yet implemented
  }

  /**
   * Shows a grid line for every y axis major tickmark.
   */
  public void setShowMajorYGrid(boolean showGrid){
    yaxis.setShowMajorGrid(showGrid);
    if(!showGrid) setShowMinorYGrid(showGrid);
  }

  /**
   * Shows a grid line for every y axis minor tickmark.
   */
  public void setShowMinorYGrid(boolean showGrid){
    // minor grids not yet implemented
  }

  public void setX(double x){}

  public void setY(double y){}

  public double getX(){return 0;}

  public double getY(){return 0;}

}
