/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.*;
import java.awt.geom.*;
import org.opensourcephysics.controls.*;

/**
 * Title:        Trail
 * Description:  A trail of pixels on the screen.  This object is often used to
 * show the path of a moving object.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */

public class Trail implements Drawable {
  private GeneralPath generalPath = new GeneralPath();
  private int numpts=0; // the number of points in the trail
  private boolean connected=true;

  public Color color=Color.black;  // changing the color is harmless so this can be public

  /**
   * Adds a point to the trail.
   * @param x double
   * @param y double
   */
  public synchronized void addPoint(double x, double y){
    if(!connected || numpts==0) generalPath.moveTo((float)x, (float)y);
    generalPath.lineTo((float)x, (float)y);
    numpts++;
  }

  /**
   * Sets the connectd flag.
   *
   * Successive points are connetected by straight lines.
   * Each point is marked as a colored pixel if the trail is not connected.
   *
   * @param connected boolean
   */
  public void setConnected(boolean connected){
    this.connected=connected;
  }

  /**
   * Gets the connected flag.
   *
   * @param connected boolean
   */
  public boolean isConnected(boolean connected) {
    return connected;
  }


  /**
   * Starts a new trail segment by moving to a new point without drawing.
   * @param x double
   * @param y double
   */
  public synchronized void moveToPoint(double x, double y) {
    generalPath.moveTo( (float) x, (float) y);
    numpts++;
  }

  /**
   * Clears all points from the trail.
   */
  public synchronized void clear(){
     numpts=0;
     generalPath.reset();
  }


/**
 * Draw the trail on the panel.
 * @param g
 */
  public void draw(DrawingPanel panel, Graphics g) {
    if(numpts==0) return;
    Graphics2D g2 = (Graphics2D)g;
    g2.setColor(color);
    // transform from world to pixel coordinates
    Shape s = generalPath.createTransformedShape(panel.getPixelTransform());
    g2.draw(s);
  }

  /**
 * Returns the XML.ObjectLoader for this class.
 *
 * @return the object loader
 */
public static XML.ObjectLoader getLoader() {
  return new Loader();
}

/**
 * A class to save and load Dataset data in an XMLControl.
 */
private static class Loader extends XMLLoader {

  public void saveObject(XMLControl control, Object obj) {
    Trail trail = (Trail) obj;
    control.setValue("connected",trail.connected);
    control.setValue("color",trail.color);
    control.setValue("number of pts",trail.numpts);
    control.setValue("general path", trail.generalPath);
  }

  public Object createObject(XMLControl control){
    return new Trail();
  }

  public Object loadObject(XMLControl control, Object obj) {
    Trail trail = (Trail) obj;
    trail.connected=control.getBoolean("connected");
    trail.color=(Color)control.getObject("color");
    trail.numpts=control.getInt("number of pts");
    trail.generalPath=(GeneralPath)control.getObject("general path");
    return obj;
  }
}

}
