package org.opensourcephysics.display;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.LayoutManager;

import javax.swing.JPanel;

/**
 * The TextPanel renders text in a component.
 * The text is surrounded by a border.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */
public class TextPanel extends JPanel {
  private static Dimension ZEROSIZE= new Dimension(0,0);
  private   String        text            = "";
  protected Font          font;
  protected String        fontname        = "TimesRoman";  //The logical name of the font to use
  protected int           fontsize        = 14;            //The font size
  protected int           fontstyle       = Font.PLAIN;    //The font style
  protected Color         textColor       = Color.black;   // text color
  protected Color         backgroundColor = Color.yellow;  // background color
  protected Dimension     dim=ZEROSIZE;


  /**
   * Constructs a TextPanel and places it within the given drawing panel.
   *
   * @param dp the drawing panel.
   */
  public TextPanel() {
    setBackground(backgroundColor);
    font         = new Font(fontname, fontstyle, fontsize);
  }

  /**
   * Constructs a TextPanel and places it within the given drawing panel.
   *
   * @param dp the drawing panel
   * @param text the text to be displayed
   */
  public TextPanel(String text) {
    this();
    setText(text);
  }


  /**
   * Sets the text.
   *
   * The position is ignored if the location is set to a corner.
   *
   * @param _text
   */
  public void setText(String _text) {
    _text = GUIUtils.parseTeX(_text);
    if(text==_text)return;
    text = _text;
    if(text==null) text="";
    Container c=this.getParent();
    if(c==null)return;
    if(c.getLayout() instanceof OSPLayout){
     ((OSPLayout)c.getLayout()).quickLayout(c,this);
     repaint();
    }else{
      c.validate();
    }
  }

  /**
   * Gets the preferred size of this component.
   * @return a dimension object indicating this component's preferred size
   * @see #getMinimumSize
   * @see LayoutManager
   */
  public Dimension getPreferredSize() {
    Container c=this.getParent();
    String text=this.text;  // local reference for thread safety
    if(c==null || text.equals("")) return ZEROSIZE;
    Graphics2D g2=(Graphics2D)c.getGraphics();
    if(g2==null) return ZEROSIZE;
    Font oldFont = g2.getFont();
    g2.setFont(font);
    FontMetrics fm        = g2.getFontMetrics();
    int         boxHeight = fm.getAscent() + 4;        // current string height
    int         boxWidth  = fm.stringWidth(text) + 6;  // current string width
    g2.setFont(oldFont);
    return new Dimension(boxWidth,boxHeight);
  }



  /**
   * Paints this panel.
   * @param g
   */
  public void paint(Graphics g) {
    String text=this.text;  // local reference for thread safety
    if (!dim.equals(getPreferredSize())) {
      dim = getPreferredSize();
      setSize(dim);
    }
    if (text.equals("") || !isVisible()) {
      return;
    }
    Graphics2D g2 = (Graphics2D) g;
    super.paintComponent(g);
    int w = getWidth();
    int h = getHeight();
    g2.setColor(textColor);
    Font oldFont = g2.getFont();
    g2.setFont(font);
    g2.drawString(text, 3, h - 4);
    g2.setFont(oldFont);
    g2.setColor(Color.black);
    g2.drawRect(0, 0, w - 1, h - 1);
  }
}
