package org.opensourcephysics.display;


/**
 * Stripchart stores data having increasing x values.
 *
 * Only data within the interval [lastx-xrange, lastx] is retained.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */
public class Stripchart extends Dataset{
  boolean rightToLeft=true;
  double xrange; // the range the independent variable
  double yrange; // the minimum range the dependent variable
  double lastx;
  boolean enabled=true;

  public Stripchart(double _xrange, double _yrange){
     super();
     xrange=Math.abs(_xrange);
     yrange=Math.abs(_yrange);
  }

  /**
   * Sets the range of the stipchart.
   * @param _xrange double
   * @param _yrange double
   */
  public void setRange(double _xrange, double _yrange){
    xrange=Math.abs(_xrange);
    yrange=Math.abs(_yrange);
  }

  public void enable(boolean enabled){
    this.enabled=enabled;
  }

  /**
   * Appends an (x,y) datum to the Stipchart.
   *
   * @param  x
   * @param  y
   */
  public void append(double x, double y) {
    if(!enabled){
      super.append(x,y);
      return;
    }
    if(index!=0 && x<lastx) clear();  //x values are not increasing so clear and restart data collection
    lastx=x;
    super.append(x,y);
    trim();
  }

  /**
   *  Appends (x,y) data-arrays to the Stipchart.
   *
   * @param  _xpoints
   * @param  _ypoints
   */
  public void append(double[] _xpoints, double[] _ypoints) {
    if (!enabled) {
      super.append(_xpoints, _ypoints);
      return;
    }
    if(index!=0 && _xpoints[0]<lastx) clear();  //new x values are not increasing so clear and restart data collection
    for(int i=1, n=_xpoints.length; i<n; i++){
      if(_xpoints[i]<_xpoints[i-1]){  // x values are not increasing so clear and return without collecting data;
         clear();
         return;
      }
    }
    lastx=_xpoints[_xpoints.length-1];
    super.append(_xpoints,_ypoints);
    trim();
  }

  /**
   *  Clears all data from this Dataset.
   */
  public void clear(){
    super.clear();
    lastx=xpoints[0];
  }

  /**
   * Trims data points whose x values are outside the xrange from the dataset.
   */
  private void trim(){
    if(index>0 && xpoints[0]<lastx-xrange){
      int counter=0;
      while(counter< index && xpoints[counter]<lastx-xrange) counter++;
      System.arraycopy(xpoints,counter,xpoints,0,index-counter);
      System.arraycopy(ypoints,counter,ypoints,0,index-counter);
      index=index-counter;
    }
    if(rightToLeft)
      xmin= lastx-xrange;
      else xmin= lastx;
    if(rightToLeft)
      xmax = lastx;
    else xmax = lastx-xrange;
    ymin=ymax=ypoints[0];
    for(int i=1; i<index; i++){
      ymin=Math.min(ymin,ypoints[i]);
      ymax=Math.max(ymax,ypoints[i]);
    }
    if(ymax-ymin<yrange){
      ymin=(ymax+ymin-yrange)/2.0;
      ymax=(ymax+ymin+yrange)/2.0;
    }
    recalculatePath();
  }

}
