package org.opensourcephysics.display;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

/**
 * Sorts table column based on their numeric values if the table's values are
 * numeric.  Otherwise sorts using toString.
 *
 * @author W. Christian
 * @version 1.0
 */
public class SortDecorator
    implements TableModel, TableModelListener {
  private TableModel realModel;
  private int indexes[];

  public SortDecorator(TableModel model) {
    if (model == null)
      throw new IllegalArgumentException(
          "null models are not allowed");
    this.realModel = model;

    realModel.addTableModelListener(this);
    allocate();
  }

  public Object getValueAt(int row, int column) {
    if (indexes.length <= row) {
      allocate();
    }
    return realModel.getValueAt(indexes[row], column);
  }

  public void setValueAt(Object aValue, int row, int column) {
    if (indexes.length <= row) {
      allocate();
    }
    realModel.setValueAt(aValue, indexes[row], column);
  }

  public void tableChanged(TableModelEvent e) {
    allocate();
  }

  public void sort(int column) {
    int rowCount = getRowCount();
    if (indexes.length <= rowCount) {
      allocate();
    }
    for (int i = 0; i < rowCount; i++) {
      for (int j = i + 1; j < rowCount; j++) {
        if (compare(indexes[i], indexes[j], column) < 0) {
          swap(i, j);
        }
      }
    }
  }

  public void swap(int i, int j) {
    int tmp = indexes[i];
    indexes[i] = indexes[j];
    indexes[j] = tmp;
  }

  public int compare(int i, int j, int column) {
    Object io = realModel.getValueAt(i, column);
    Object jo = realModel.getValueAt(j, column);
    if (io instanceof Integer) {
      int a = ( (Integer) io).intValue();
      int b = ( (Integer) jo).intValue();
      return (b < a) ? -1 : ( (b > a) ? 1 : 0);
    }
    if (io instanceof Double) {
      double a = ( (Double) io).doubleValue();
      double b = ( (Double) jo).doubleValue();
      return (b < a) ? -1 : ( (b > a) ? 1 : 0);
    }

    int c = jo.toString().compareTo(io.toString());
    return (c < 0) ? -1 : ( (c > 0) ? 1 : 0);
  }

  private void allocate() {
    indexes = new int[getRowCount()];

    for (int i = 0; i < indexes.length; ++i) {
      indexes[i] = i;
    }
  }

  public int getRowCount() {
    return realModel.getRowCount();
  }

  public int getColumnCount() {
    return realModel.getColumnCount();
  }

  public String getColumnName(int columnIndex) {
    return realModel.getColumnName(columnIndex);
  }

  public Class getColumnClass(int columnIndex) {
    return realModel.getColumnClass(columnIndex);
  }

  public boolean isCellEditable(int rowIndex, int columnIndex) {
    return realModel.isCellEditable(rowIndex, columnIndex);
  }

  public void addTableModelListener(TableModelListener l) {
    realModel.addTableModelListener(l);
  }

  public void removeTableModelListener(TableModelListener l) {
    realModel.removeTableModelListener(l);
  }
}
