/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2002  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferStrategy;
import java.io.File;
import javax.swing.JApplet;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.WindowConstants;
import javax.swing.JMenu;

import javax.swing.JMenuBar;
import javax.swing.JMenuItem;

/**
 * OSPFrame is the base class for Open Source Physics JFrames such as DrawingFrame and DataTableFrame.
 *
 * Copyright:    Copyright (c) 2002
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class OSPFrame extends JFrame {

   static int topx = 10;
   static int topy = 100;

   /** Set to <I>true</I> if a program should automatically render this frame after every animation step. */
   protected boolean animated = false;

   /** Set to <I>true</I> if a program should automatically clear the data when an animation is initialized. */
   protected boolean autoclear = false;

   /** Set <I>true</I> if the program is an applet. */
   public static boolean appletMode = false;

   /**
    * Field myApplet provides a static reference to an applet context
    * so that the document base and code base can be obtained in applet mode.
    */
   public static JApplet applet;

   /** The thread group that created this object. */
   public ThreadGroup constructorThreadGroup = Thread.currentThread().getThreadGroup();
   protected boolean keepHidden = false;
   protected BufferStrategy strategy;
   private static JFileChooser chooser;

   /**
    * OSPFrame constructor with a title.
    */
   public OSPFrame(String title) {
      super(GUIUtils.parseTeX(title));
      if(appletMode) {
         keepHidden = true;
      }
      setLocation(topx, topy);
      Dimension d = java.awt.Toolkit.getDefaultToolkit().getScreenSize();
      topx = Math.min(topx+20, (int) d.getWidth()-100);
      topy = Math.min(topy+20, (int) d.getHeight()-100);
      setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
   }

   /**
    * OSPFrame constructor.
    */
   public OSPFrame() {
      this("Open Source Physics");
   }

   /**
    * OSPFrame constructor with a new content pane.
    */
   public OSPFrame(Container contentPane) {
      this();
      setContentPane(contentPane);
   }

   /**
    * Sets the title for this frame to the specified string after converting TeX math symbols to characters.
    * @param title the title to be displayed in the frame's border.
    *              A <code>null</code> value
    *              is treated as an empty string, "".
    * @see      #getTitle
    */
   public void setTitle(String title) {
      super.setTitle(GUIUtils.parseTeX(title));
   }

   /**
    * Gets the ICONIFIED flag for this frame.
    *
    * @return boolean true if frame is iconified; false otherwise
    */
   public boolean isIconified() {
      return(getExtendedState()&ICONIFIED)==1;
   }

   /**
    * Reads the animated property.
    *
    * @return boolean
    */
   public boolean isAnimated() {
      return animated;
   }

   /**
    * Sets the animated property.
    *
    * @param animated
    */
   public void setAnimated(boolean animated) {
      this.animated = animated;
   }

   /**
    * Reads the animated property.
    *
    * @return boolean
    */
   public boolean isAutoclear() {
      return autoclear;
   }

   /**
    * Sets the autoclear property.
    *
    * @param autoclear
    */
   public void setAutoclear(boolean autoclear) {
      this.autoclear = autoclear;
   }

   /**
    * Clears data from drawing objects within this frame.
    *
    * The default method does nothing.
    * Override this method to select the object(s) and the data to be cleared.
    */
   public void clearData() {}

   public void setSize(int width, int height) {
      super.setSize(width, height);
      validate();
   }

   /**
    * Shows the frame on the screen if the keep hidden flag is false.
    *
    * @deprecated
    */
   public void show() {
      if(!keepHidden) {
         super.show();
      }
   }

   /**
    * Shows or hides this component depending on the value of parameter
    * <code>b</code> and the <code>keepHidden</code> flag.
    *
    * OSP Applets often keep windows hidden.
    *
    * @param b
    */
   public void setVisible(boolean b) {
      if(!keepHidden) {
         super.setVisible(b);
      }
   }

   /**
    * Sets the keepHidden flag.
    *
    * @param _keepHidden
    */
   public void setKeepHidden(boolean _keepHidden) {
      keepHidden = _keepHidden;
      if(keepHidden) {
         setVisible(false);
      }
   }

   /**
    * Reads the keepHidden flag.
    *
    * @param _keepHidden
    */
   public boolean isKeepHidden() {
      return keepHidden;
   }

   /**
    * Gets the ThreadGroup that constructed this frame.
    *
    * @return the ThreadGroup
    */
   public ThreadGroup getConstructorThreadGroup() {
      return constructorThreadGroup;
   }

   /**
    * Creates a BufferStrategy based on the capabilites of the hardware.
    */
   public void createBufferStrategy() {
      createBufferStrategy(2);
      strategy = this.getBufferStrategy();
   }

   /**
    * Shows (repaints) the frame useing the current BufferStrategy.
    */
   public void bufferStrategyShow() {
      if((strategy)==null) {
         createBufferStrategy();
      }
      if(isIconified()||!isShowing()) {
         return;
      }
      Graphics g = strategy.getDrawGraphics();
      paintComponents(g);
      g.dispose();
      strategy.show();
   }

   /**
    * Renders the frame.  Subclass this method to render the contents of this frame in the calling thread.
    */
   public void render() {}

   /**
    * Gets a menu with the given name from the menu bar.  Returns null if menu item does not exist.
    *
    * @param menuName String
    * @return JMenu
    */
   public JMenu getMenu(String menuName) {
      JMenuBar menuBar = getJMenuBar();
      if(menuBar==null) {
         return null;
      }
      menuName = menuName.trim();
      JMenu menu = null;
      for(int i = 0; i<menuBar.getMenuCount(); i++) {
         JMenu next = menuBar.getMenu(i);
         if(next.getText().trim().equals(menuName)) {
            menu = next;
            break;
         }
      }
      return menu;
   }

   /**
    * Removes a menu with the given name from the menu bar and returns the removed item.
    * Returns null if menu item does not exist.
    *
    * @param menuName String
    * @return JMenu
    */
   public JMenu removeMenu(String menuName) {
      JMenuBar menuBar = getJMenuBar();
      if(menuBar==null) {
         return null;
      }
      menuName = menuName.trim();
      JMenu menu = null;
      for(int i = 0; i<menuBar.getMenuCount(); i++) {
         JMenu next = menuBar.getMenu(i);
         if(next.getText().trim().equals(menuName)) {
            menu = next;
            menuBar.remove(i);
            break;
         }
      }
      return menu;
   }

   /**
    * Removes a menu item with the given name from the menu bar and returns the removed item.
    * Returns null if menu item does not exist.
    *
    * @param menuName String
    * @return JMenu
    */
   public JMenuItem removeMenuItem(String menuName, String itemName) {
      JMenu menu = getMenu(menuName);
      if(menu==null) {
         return null;
      }
      itemName = itemName.trim();
      JMenuItem item = null;
      for(int i = 0; i<menu.getItemCount(); i++) {
         JMenuItem next = menu.getItem(i);
         if(next.getText().trim().equals(itemName)) {
            item = next;
            menu.remove(i);
            break;
         }
      }
      return item;
   }

   /**
    * Gets a file chooser.
    *
    * The choose is static and will therefore be the same for all OSPFrames.
    *
    * @return the chooser
    */
   public static JFileChooser getChooser() {
      if(chooser!=null) {
         return chooser;
      }
      chooser = new JFileChooser(new File("."+File.separatorChar));
      //chooser = new JFileChooser();  // obtains a chooser at the home directory
      javax.swing.filechooser.FileFilter defaultFilter = chooser.getFileFilter();
      javax.swing.filechooser.FileFilter xmlFilter = new javax.swing.filechooser.FileFilter() {

         // accept all directories and *.xml files.
         public boolean accept(File f) {
            if(f==null) {
               return false;
            }
            if(f.isDirectory()) {
               return true;
            }
            String extension = null;
            String name = f.getName();
            int i = name.lastIndexOf('.');
            if((i>0)&&(i<name.length()-1)) {
               extension = name.substring(i+1).toLowerCase();
            }
            if((extension!=null)&&(extension.equals("xml"))) {
               return true;
            }
            return false;
         }

         // the description of this filter
         public String getDescription() {
            return "XML files";
         }
      };
      javax.swing.filechooser.FileFilter txtFilter = new javax.swing.filechooser.FileFilter() {

         // accept all directories and *.txt files.
         public boolean accept(File f) {
            if(f==null) {
               return false;
            }
            if(f.isDirectory()) {
               return true;
            }
            String extension = null;
            String name = f.getName();
            int i = name.lastIndexOf('.');
            if((i>0)&&(i<name.length()-1)) {
               extension = name.substring(i+1).toLowerCase();
            }
            if((extension!=null)&&extension.equals("txt")) {
               return true;
            }
            return false;
         }

         // the description of this filter
         public String getDescription() {
            return "TXT files";
         }
      };
      chooser.addChoosableFileFilter(xmlFilter);
      chooser.addChoosableFileFilter(txtFilter);
      chooser.setFileFilter(defaultFilter);
      return chooser;
   }
}
