/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;

/**
 * MeasuredImage contains an image and a scale in world units.
 *
 * When a MeasuredImage is added to a drawing panel, the image will scale itself to
 * the panel's world units.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */
public class MeasuredImage implements Measurable {
   protected BufferedImage image;
   protected double xmin,xmax,ymin,ymax;
   protected boolean visible=true;

   /**
    * Constructs a MeasuredImage with a pixel scale.
    *
    * @param img the image
    */
   public MeasuredImage() {
     this(null, 0, 0, 0, 0);
   }

   /**
    * Constructs a MeasuredImage with a pixel scale.
    *
    * @param img the image
    */
   public MeasuredImage (BufferedImage image) {
     this(image, 0, image.getWidth(), 0, image.getHeight());
   }
 /**
  * Constructs a MeasuredImage with the given scale.
  *
  * @param _image
  * @param _xmin
  * @param _xmax
  * @param _ymin
  * @param _ymax
  */
   public MeasuredImage (BufferedImage _image, double _xmin, double _xmax, double _ymin, double _ymax) {
     image=_image;
     xmin=_xmin;
     xmax=_xmax;
     ymin=_ymin;
     ymax=_ymax;
   }

   public void setImage(BufferedImage _image){
     image=_image;
   }


   /**
    * Sets the visibility of the lattice.
    * Drawing will be disabled if visible is false.
    *
    * @param isVisible
    */
   public void setVisible(boolean isVisible){
     visible=isVisible;
   }

  /**
   * Draws the image on the panel.
   *
   * @param panel
   * @param g
   */
  public void draw(DrawingPanel panel, Graphics g) {
    if(!visible) return;
    if(image == null) {
      panel.setMessage("No image");
      return;
    }
    Graphics2D g2 = (Graphics2D)g;
    AffineTransform gat = g2.getTransform();    // save graphics transform
    double sx=(xmax-xmin)*panel.xPixPerUnit/image.getWidth();
    double sy=(ymax-ymin)*panel.yPixPerUnit/image.getHeight();

    g2.transform(AffineTransform.getTranslateInstance(
          panel.leftGutter+panel.xPixPerUnit*(xmin-panel.xmin),
          panel.topGutter+panel.yPixPerUnit*(panel.ymax-ymax)));
    g2.transform(AffineTransform.getScaleInstance(sx,sy));
    g2.drawImage(image, 0 , 0, panel);
    g2.setTransform(gat);                       // restore graphics transform
  }


   public boolean isMeasured () {
      if(image==null) return false;
      return true;
   }

   public double getXMin () {
      return xmin;
   }

   public double getXMax () {
      return xmax;
   }

   public double getYMin () {
      return ymin;
   }

   public double getYMax () {
      return ymax;
   }

   public void setXMin (double _xmin) {
     xmin=_xmin;
   }

   public void setXMax (double _xmax) {
      xmax=_xmax;
   }

   public void setYMin (double _ymin) {
      ymin=_ymin;
   }

   public void setYMax (double _ymax) {
      ymax=_ymax;
   }

   public void setMinMax(double _xmin, double _xmax, double _ymin, double _ymax){
      xmin=_xmin;
      xmax=_xmax;
      ymin=_ymin;
      ymax=_ymax;
   }
}
