/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.*;
import java.awt.geom.*;

import org.opensourcephysics.controls.*;

/**
 * A shape that implements Interactive.
 * @author Wolfgang Christian
 * @version 1.0
 */
public class InteractiveShape extends AbstractInteractive implements Measurable {
   // fillColor uses color in the superclass
  public Color edgeColor = Color.red;  // the edge color
  protected Shape shape;
  protected String shapeClass;
  protected double theta;
  protected double width, height;  // an estimate of the shape's width and height
  protected double xoff, yoff;  // offset from center
  protected boolean pixelSized = false;  // width and height are fixed and given in pixels
  AffineTransform toPixels = new AffineTransform();
  boolean enableMeasure = false;  // enables the measure so that this object affects a drawing panel's scale

  /**
   * Constructs an InteractiveShape with the given coordinates.
   *
   * @param x coordinate
   * @param y coordinate
   */
  public InteractiveShape(Shape s, double _x, double _y) {
    color = new Color(255, 128, 128, 128);  // transparent light red fill color
    shape = s;
    x = _x;
    y = _y;
    if(shape==null) {
      return;
    }
    Rectangle2D bounds = shape.getBounds2D();
    width = bounds.getWidth();
    height = bounds.getHeight();
    shapeClass = shape.getClass().getName();
    shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
  }

  /**
   * Constructs an InteractiveShape at the origin.
   */
  public InteractiveShape(Shape s) {
    this(s, 0, 0);
  }

  /**
   * Creates an interactive ellipse.
   *
   * @param x
   * @param y
   * @param w
   * @param h
   * @return
   */
  public static InteractiveShape createEllipse(double x, double y, double w, double h) {
    Shape shape = new Ellipse2D.Double(-w/2, -h/2, w, h);
    InteractiveShape is = new InteractiveShape(shape, x, y);
    is.width = w;
    is.height = h;
    return is;
  }

  /**
   * Creates an interactive circle.
   *
   * @param x
   * @param y
   * @param d the diameter
   * @return the interactive circle
   */
  public static InteractiveShape createCircle(double x, double y, double d){
     return createEllipse(x, y, d, d);
  }

  /**
   * Creates an interactive rectangle.
   * @param x
   * @param y
   * @param w
   * @param h
   * @return the interactive rectangle
   */
  public static InteractiveShape createRectangle(double x, double y, double w, double h) {
    Shape shape = new Rectangle2D.Double(-w/2, -h/2, w, h);
    InteractiveShape is = new InteractiveShape(shape, x, y);
    is.width = w;
    is.height = h;
    return is;
  }

  /**
   * Creates an interactive triangle with a base parallel to the x axis.
   *
   * @param x
   * @param y
   * @param b  base
   * @param h  height
   * @return the interactive triangle
   */
  public static InteractiveShape createTriangle(double x, double y, double b, double h) {
    GeneralPath path = new GeneralPath();
    path.moveTo((float) (-b/2), (float) (-h/2));
    path.lineTo((float) (+b/2), (float) (-h/2));
    path.lineTo((float) 0, (float) (h/2));
    path.closePath();
    Shape shape = path;
    InteractiveShape is = new InteractiveShape(shape, x, y);
    is.width = b;
    is.height = h;
    return is;
  }

  /**
   * Creates an interactive image.
   * @param x
   * @param y
   * @param image
   * @return the rectangle
   */
  public static InteractiveShape createImage(Image image, double x, double y) {
    InteractiveImage is= new InteractiveImage(image, x, y);
    return is;
  }

  /**
 * Creates an interactive image.
 * @param x
 * @param y
 * @param image
 * @return the rectangle
 */
public static InteractiveShape createTextLine(double x, double y, String text) {
  InteractiveTextLine is= new InteractiveTextLine(text, x, y);
  return is;
}


  /**
   * Creates an interactive arrow.
   * @param x
   * @param y
   * @param b base
   * @param h height
   * @return the arrow
   */
  public static InteractiveShape createArrow(double x, double y, double w, double h) {
    InteractiveArrow is=new InteractiveArrow(x, y, w, h);
    is.setHeightDrag(false);
    is.setWidthDrag(false);
    is.hideBounds=true;
    return is;
  }

  /**
   * Creates an interactive arrow.
   * @param x
   * @param y
   * @param b base
   * @param h height
   * @return the arrow
   */
  public static InteractiveShape createCenteredArrow(double x, double y, double w, double h) {
    InteractiveCenteredArrow is= new InteractiveCenteredArrow(x, y, w, h);
    is.setHeightDrag(false);
    is.setWidthDrag(false);
    is.hideBounds=true;
    return is;
  }

  /**
   * Creates an interactive square.
   * @param x
   * @param y
   * @param w
   * @return the interactive square
   */
  public static InteractiveShape createSquare(double x, double y, double w) {
    Shape shape = new Rectangle2D.Double(-w/2, -w/2, w, w);
    return new InteractiveShape(shape, x, y);
  }

  /**
   * Transforms the shape.
   *
   * @param transformation AffineTransform
   */
  public void transform(AffineTransform transformation){
     shape = transformation.createTransformedShape(shape);
  }


  /**
   * Draws the shape.
   *
   * @param panel the drawing panel
   * @param g  the graphics context
   */
  public void draw(DrawingPanel panel, Graphics g) {
    Graphics2D g2 = ((Graphics2D) g);
    toPixels = panel.getPixelTransform();
    Shape temp;
    if(pixelSized) {
      Point2D pt = new Point2D.Double(x, y);
      pt = toPixels.transform(pt, pt);
      // translate the shape to correct pixel coordinates
      temp = new AffineTransform(1, 0, 0, -1, -x+pt.getX()+xoff, y+pt.getY()-yoff).createTransformedShape(shape);
      temp = AffineTransform.getRotateInstance(-theta, pt.getX(), pt.getY()).createTransformedShape(temp);
    } else {
      temp = toPixels.createTransformedShape(shape);
    }
    g2.setPaint(color);
    g2.fill(temp);
    g2.setPaint(edgeColor);
    g2.draw(temp);
  }

  /**
   * Tests if the specified coordinates are inside the boundary of the
   * <code>Shape</code>.
   * @param x,&nbsp;y the specified coordinates
   * @return <code>true</code> if the specified coordinates are inside
   *         the <code>Shape</code> boundary; <code>false</code>
   *         otherwise.
   */
  public boolean contains(double x, double y) {
    if(shape.contains(x, y)) {
      return true;
    }
    return false;
  }

  /**
   * Gets the Java shape that is being drawn.
   * @return the shape
   */
  public Shape getShape() {
    return shape;
  }

  /**
   * Transforms the shape using the given matrix.
   * @param mat double[][]
   */
  public void tranform(double[][] mat) {
    shape = (new AffineTransform(mat[0][0], mat[1][0], mat[0][1], mat[1][1], mat[0][2], mat[1][2])).createTransformedShape(shape);
  }

  /**
   * Determines if the shape is enabled and if the given pixel coordinates are within the shape.
   *
   * @param panel DrawingPanel
   * @param xpix int
   * @param ypix int
   * @return boolean
   */
  public boolean isInside(DrawingPanel panel, int xpix, int ypix) {
    if(shape==null || !enabled) {
      return false;
    }
    if(shape.contains(panel.pixToX(xpix), panel.pixToY(ypix))) {
      return true;
    }
    return false;
  }

  /**
   * Sets the shape's drawing colors.
   *
   * The error bar color is set equal to the edge color.
   *
   * @param  _fillColor
   * @param  _edgeColor
   */
  public void setMarkerColor(Color _fillColor, Color _edgeColor) {
    color = _fillColor;
    edgeColor = _edgeColor;
  }

  /**
   * Sets the rotation angle in radians.
   *
   * @param theta the new angle
   */
  public void setTheta(double theta) {
    if(!pixelSized) {
      shape = AffineTransform.getRotateInstance(theta-this.theta, x, y).createTransformedShape(shape);
    }
    this.theta = theta;
  }

  /**
   * Sets the pixelSized flag.
   *
   * Pixel sized shapes use pixels for width and height.
   *
   * @param enable boolean
   */
  public void setPixelSized(boolean enable) {
    this.pixelSized = enable;
  }

  /**
   * Gets the width of this shape.
   *
   * @return double
   */
  public double getWidth() {
    return width;
  }

  /**
   * Sets the width of the shape to the given value.
   *
   * @param width double
   */
  public void setWidth(double width) {
    width = Math.abs(width);
    double w = width/this.width;
    if(w<0.02) {
      return;
    }
    if(pixelSized) {
      shape = AffineTransform.getTranslateInstance(-x, -y).createTransformedShape(shape);
      shape = AffineTransform.getScaleInstance(w, 1).createTransformedShape(shape);
      shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
    } else {
      shape = AffineTransform.getTranslateInstance(-x, -y).createTransformedShape(shape);
      shape = AffineTransform.getRotateInstance(-theta).createTransformedShape(shape);
      shape = AffineTransform.getScaleInstance(w, 1).createTransformedShape(shape);
      shape = AffineTransform.getRotateInstance(theta).createTransformedShape(shape);
      shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
    }
    xoff *= w;
    this.width = width;
  }

  /**
   * Gets the height of this shape.
   *
   * @return double
   */
  public double getHeight() {
    return height;
  }

  /**
   * Sets the height of the shape to the given value.
   * @param height double
   */
  public void setHeight(double height) {
    height = Math.abs(height);
    double h = height/this.height;
    if(h<0.02) {
      return;
    }
    if(pixelSized) {
      shape = AffineTransform.getTranslateInstance(-x, -y).createTransformedShape(shape);
      shape = AffineTransform.getScaleInstance(1, h).createTransformedShape(shape);
      shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
    }else {
      shape = AffineTransform.getTranslateInstance(-x, -y).createTransformedShape(shape);
      shape = AffineTransform.getRotateInstance(-theta).createTransformedShape(shape);
      shape = AffineTransform.getScaleInstance(1, h).createTransformedShape(shape);
      shape = AffineTransform.getRotateInstance(theta).createTransformedShape(shape);
      shape = AffineTransform.getTranslateInstance(x, y).createTransformedShape(shape);
    }
    yoff *= h;
    this.height = height;
  }

  /**
   * Sets the drawing offset;
   *
   * Fixed size shapes cannot be offset.
   *
   * @param xoffset double
   * @param yoffset double
   */
  public void setOffset(double xoffset, double yoffset) {
    if(!pixelSized) {  //change the actual shape
      shape = AffineTransform.getTranslateInstance(x+xoffset, y+yoffset).createTransformedShape(shape);
    }
    xoff = xoffset;
    yoff = yoffset;
  }

  /**
   * Sets the x and y coordinates.
   *
   * @param _x
   * @param _y
   */
  public void setXY(double _x, double _y) {
    shape = AffineTransform.getTranslateInstance(_x-x, _y-y).createTransformedShape(shape);
    x = _x;
    y = _y;
  }

  /**
   * Sets the x coordinate.
   *
   * @param x
   */
  public void setX(double _x) {
    shape = AffineTransform.getTranslateInstance(_x-x, 0).createTransformedShape(shape);
    x = _x;
  }

  /**
   * Sets the y coordinate.
   *
   * @param y
   */
  public void setY(double _y) {
    shape = AffineTransform.getTranslateInstance(0, _y-y).createTransformedShape(shape);
    y = _y;
  }

  /**
   * Gets a description of this object.
   * @return String
   */
  public String toString() {
    return "InteractiveShape:"+"\n \t shape="+shapeClass+"\n \t x="+x+"\n \t y="+y+"\n \t width="+width+"\n \t height="+height+"\n \t theta="+theta;
  }

  /**
   * Enables the measured flag so that this arrow effects the scale of a drawing panel.
   *
   * @return minimum
   */
  public void setMeasured(boolean _enableMeasure) {
    enableMeasure = _enableMeasure;
  }

  /**
   * Determines if this circle should effect the scale of a drawing panel.
   *
   * @return minimum
   */
  public boolean isMeasured() {
    return enableMeasure;
  }

  /**
   * Implements measurable by getting the x center of the circle.
   *
   * @return minimum
   */
  public double getXMin() {
    if(pixelSized) {
      return x-width/toPixels.getScaleX()/2;
    }
    return shape.getBounds2D().getX();
  }

  /**
   * Implements measurable by getting the x center of the circle.
   *
   * @return maximum
   */
  public double getXMax() {
    if(pixelSized) {
      return x+width/toPixels.getScaleX()/2;
    }
    return shape.getBounds2D().getX()+shape.getBounds2D().getWidth();
  }

  /**
   * Implements measurable by getting the y center of the circle.
   *
   * @return minimum
   */
  public double getYMin() {
    if(pixelSized) {
      return y-height/toPixels.getScaleY()/2;
    }
    return shape.getBounds2D().getY();
  }

  /**
   * Implements measurable by getting the y center of the circle.
   *
   * @return maximum
   */
  public double getYMax() {
    if(pixelSized) {
      return y+height/toPixels.getScaleY()/2;
    }
    return shape.getBounds2D().getY()+shape.getBounds2D().getHeight();
  }

  /**
   * Gets the XML object loader for this class.
   * @return ObjectLoader
   */
  public static XML.ObjectLoader getLoader() {
    return new InteractiveShapeLoader();
  }

   /**
   * A class to save and load InteractiveShape in an XMLControl.
   */
  protected static class InteractiveShapeLoader extends XMLLoader {
    public void saveObject(XMLControl control, Object obj) {
      InteractiveShape interactiveShape = (InteractiveShape) obj;
      control.setValue("geometry", interactiveShape.shapeClass);
      control.setValue("x", interactiveShape.x);
      control.setValue("y", interactiveShape.y);
      control.setValue("width", interactiveShape.width);
      control.setValue("height", interactiveShape.height);
      control.setValue("x offset", interactiveShape.xoff);
      control.setValue("y offset", interactiveShape.yoff);
      control.setValue("theta", interactiveShape.theta);
      control.setValue("pixel sized", interactiveShape.pixelSized);
      control.setValue("is enabled", interactiveShape.isEnabled());
      control.setValue("is measured", interactiveShape.isMeasured());
      control.setValue("color", interactiveShape.color);
      Shape shape=AffineTransform.getRotateInstance(-interactiveShape.theta,interactiveShape.x,interactiveShape.y).createTransformedShape(interactiveShape.shape);
      control.setValue("general path", (GeneralPath)shape);
    }

    public Object createObject(XMLControl control) {
      return new InteractiveShape(new Rectangle2D.Double(0, 0, 0, 0));  // default shape is a rectangle for now
    }

    protected Shape getShape(String type, double x, double y, double w, double h) {
      if(type.equals(Ellipse2D.Double.class.getName())) {
        return new Ellipse2D.Double(x-w/2, y-h/2, w, h);
      } else if(type.equals(Rectangle2D.Double.class.getName())) {
        return new Rectangle2D.Double(x-w/2, y-h/2, w, h);
      } else {
        return null;
      }
    }

    public Object loadObject(XMLControl control, Object obj) {
      InteractiveShape interactiveShape = (InteractiveShape) obj;
      String type = control.getString("geometry");
      double x = control.getDouble("x");
      double y = control.getDouble("y");
      double theta = control.getDouble("theta");
      Shape shape = getShape(type, x, y, control.getDouble("width"), control.getDouble("height"));
      if(shape == null){  // check for special geometry
        interactiveShape.shape=(GeneralPath)control.getObject("general path");
      }else{
        interactiveShape.shape = shape;
      }
      // the shape should already be scaled so just set the instatance fields
      interactiveShape.width = control.getDouble("width");
      interactiveShape.height = control.getDouble("height");
      interactiveShape.xoff = control.getDouble("x offset");
      interactiveShape.yoff = control.getDouble("y offset");
      interactiveShape.x=x;
      interactiveShape.y=y;
      interactiveShape.setPixelSized(control.getBoolean("pixel sized"));
      interactiveShape.setEnabled(control.getBoolean("is enabled"));
      interactiveShape.setMeasured(control.getBoolean("is measured"));
      interactiveShape.color = (Color) control.getObject("color");
      interactiveShape.setTheta(theta); // orient the shape
      return obj;
    }

    static { // needs this loader
      XML.setLoader(java.awt.geom.GeneralPath.class, new GeneralPathLoader());
    }

  }
}
