/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;
import java.awt.Cursor;
import java.awt.event.MouseEvent;
import java.text.DecimalFormat;

/**
 * InteractivePanel is a drawing panel that invokes the handleMouseAction method in
 * Interactive objects.
 * @author Wolfgang Christian
 * @author Francisco Equembre
 * @version 1.0
 */
public class InteractivePanel extends DrawingPanel implements InteractiveMouseHandler {

   public static final int MOUSE_PRESSED = 1;
   public static final int MOUSE_RELEASED = 2;
   public static final int MOUSE_DRAGGED = 3;
   public static final int MOUSE_CLICKED = 4;
   public static final int MOUSE_ENTERED = 5;
   public static final int MOUSE_EXITED = 6;
   public static final int MOUSE_MOVED = 7;
   protected boolean containsInteractive = false;
   protected int mouseAction = 0;
   protected MouseEvent mouseEvent = null;
   protected InteractiveMouseHandler interactive = null;
   private Interactive iaDragable = null;  // interactive object that is being dragged
   private Selectable iaSelectable = null; // interactive object that has been selected

   /**
    * Constructs an InteractivePanel with the given handler.
    * @param in InteractiveMouseHandler
    */
   public InteractivePanel(InteractiveMouseHandler in) {
      this();
      interactive = in;
   }

   /**
    * Constructs an InteractivePanel with an internal handler.
    */
   public InteractivePanel() {
      // remove the drawing panel mouse controller
      removeMouseListener(mouseController);
      removeMouseMotionListener(mouseController);
      // create and add a new mouse controller for interactive drawing
      mouseController = new IADMouseController();
      addMouseListener(mouseController);
      addMouseMotionListener(mouseController);
      interactive = this; // this panel is the default handler
   }

   /**
    * Adds a drawable object to the drawable list.
    * @param drawable
    */
   public synchronized void addDrawable(Drawable drawable) {
      super.addDrawable(drawable);
      if(drawable instanceof Interactive) {
         containsInteractive = true;
      }
   }

   /**
    * Removes all drawable objects from the drawable list.
    */
   public synchronized void clear() {
      super.clear();
      containsInteractive = false;
   }

   /**
    * Sets the interactive mouse handler.
    *
    * The interactive mouse handler is notified whenever a mouse action occurs.
    *
    * @param handler the mouse handler
    */
   public void setInteractiveMouseHandler(InteractiveMouseHandler handler) {
      interactive = handler;
   }

   /**
    * Handles mouse actions by dragging the current interactive drawable object.
    *
    * @param panel
    * @param evt
    */
   public void handleMouseAction(InteractivePanel panel, MouseEvent evt) {
      switch(panel.getMouseAction()) {
         case InteractivePanel.MOUSE_CLICKED :
            Interactive clickedIA = getInteractive();
            if((panel.getMouseClickCount()<2)||(clickedIA==null)||!(clickedIA instanceof Selectable)) {
               return;
            }
            if((iaSelectable!=null)&&(iaSelectable!=clickedIA)) {
               iaSelectable.setSelected(false);
            }
            iaSelectable = ((Selectable) clickedIA);
            iaSelectable.toggleSelected();
            validImage=false;
            if(!getIgnoreRepaint())panel.repaint();
            break;
         case InteractivePanel.MOUSE_DRAGGED :
            if(iaDragable==null) {
               return;              // nothing to drag
            }
            double x = panel.getMouseX();
            double y = panel.getMouseY();
            if(evt.getX()<1+leftGutter) {
               x = panel.pixToX(1+leftGutter);
            }
            if(evt.getX()>panel.getWidth()-1-rightGutter) {
               x = panel.pixToX(panel.getWidth()-1-rightGutter);
            }
            if(evt.getY()<1+topGutter) {
               y = panel.pixToY(1+topGutter);
            }
            if(evt.getY()>panel.getHeight()-1-bottomGutter) {
               y = panel.pixToY(panel.getHeight()-1-bottomGutter);
            }
            iaDragable.setXY(x, y); // drag the interactive object
            validImage=false;
            if(!getIgnoreRepaint())panel.repaint();        // repaint to keep the screen up to date
            break;
      }
   }

   /**
    * Gets the interactive object that was accessed by the last mouse event.
    * @return
    */
   public Interactive getInteractive() {
      if(!containsInteractive) {
         return null; // don't check unless we have a least one Interactive
      }
      if(iaDragable!=null) {
         return iaDragable;
      }
      if((iaSelectable!=null)&&iaSelectable.isSelected()) {
         // check only selected object
         Interactive iad = ((Interactive) iaSelectable).findInteractive(this, mouseEvent.getX(), mouseEvent.getY());
         return iad;
      }
      Object[] array = drawableList.toArray();
      for(int i = array.length-1; i>=0; i--) {
         Object obj = array[i];
         if(obj instanceof Interactive) {
            Interactive iad = ((Interactive) obj).findInteractive(this, mouseEvent.getX(), mouseEvent.getY());
            if(iad!=null) {
               return iad;
            }
         }
      }
      return null;
   }

   /**
    * Shows the coordinates in the text box in the lower left hand corner.
    *
    * @param show
    */
   public void setShowCoordinates(boolean show) {
      showCoordinates = show;
   }

   /**
    * Gets the mouse button of the last mouse event.
    * @return int
    */
   public int getMouseButton() {
      switch(mouseEvent.getModifiers()) {
         case java.awt.event.InputEvent.BUTTON1_MASK :
            return 1;
         case java.awt.event.InputEvent.BUTTON2_MASK :
            return 2;
         case java.awt.event.InputEvent.BUTTON3_MASK :
            return 3;
         default :
            return 0;
      }
   }

   /**
    * Gets the click count of the last mouse event.
    *
    * @return int
    */
   public int getMouseClickCount() {
      return mouseEvent.getClickCount();
   }

   /**
    * Gets the last mouse action.
    * @return int
    */
   public int getMouseAction() {
      return mouseAction;
   }

   /**
    * Gets the x pixel coordinate of the last mouse event.
    * @return
    */
   public int getMouseIntX() {
      return mouseEvent.getX();
   }

   /**
    * Gets the y pixel coordinate of the last mouse event.
    * @return
    */
   public int getMouseIntY() {
      return mouseEvent.getY();
   }

   /**
    * Gets the x world coordinate of the last mouse event.
    * @return
    */
   public double getMouseX() {
      return pixToX(mouseEvent.getX());
   }

   /**
    * Gets the y world coordinate of the last moust event
    * @return
    */
   public double getMouseY() {
      return pixToY(mouseEvent.getY());
   }

   /**
    * Saves the last mouse event.
    * @param type
    * @param evt
    */
   public void saveMouseEvent(int type, java.awt.event.MouseEvent evt) {
      mouseAction = type;
      mouseEvent = evt;
   }

   /**
    * The inner class that will handle all mouse related events.
    */
   private class IADMouseController extends MouseController {

      protected DecimalFormat scientificFormat = new DecimalFormat("0.###E0");
      protected DecimalFormat decimalFormat = new DecimalFormat("0.00");

      /**
       * Handle the mouse pressed event.
       * @param e
       */
      public void mousePressed(MouseEvent e) {
         mouseEvent = e;
         mouseAction = MOUSE_PRESSED;
         if(interactive!=null) { // is there an object available to hande the mouse event
            interactive.handleMouseAction(InteractivePanel.this, e);
            iaDragable = null;   // force the panel to search all drawables
            iaDragable = getInteractive();
            if(iaDragable!=null) {
               if(iaDragable instanceof Selectable) {
                  setMouseCursor(((Selectable) iaDragable).getPreferredCursor());
               } else {
                  setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
               }
            }
         }
         if(showCoordinates) {
            String s = coordinateStrBuilder.getCoordinateString(InteractivePanel.this, e);
            blMessageBox.setText(s);
         }
      }

      /**
       * Handles the mouse released event.
       * @param e
       */
      public void mouseReleased(MouseEvent e) {
         mouseEvent = e;
         mouseAction = MOUSE_RELEASED;
         if(interactive!=null) {
            interactive.handleMouseAction(InteractivePanel.this, e);
         }
         iaDragable = null;
         if(showCoordinates) {
            blMessageBox.setText(null);
         }
         setMouseCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
      }

      /**
       * Handles the mouse entered event.
       * @param e
       */
      public void mouseEntered(MouseEvent e) {
         if(showCoordinates) {
            setMouseCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
         }
         mouseEvent = e;
         mouseAction = MOUSE_ENTERED;
         if(interactive!=null) {
            interactive.handleMouseAction(InteractivePanel.this, e);
         }
      }

      /**
       * Handles the mouse exited event.
       * @param e
       */
      public void mouseExited(MouseEvent e) {
         setMouseCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
         mouseEvent = e;
         mouseAction = MOUSE_EXITED;
         if(interactive!=null) {
            interactive.handleMouseAction(InteractivePanel.this, e);
         }
      }

      /**
       * Handles the mouse clicked event.
       * @param e
       */
      public void mouseClicked(MouseEvent e) {
         mouseEvent = e;
         mouseAction = MOUSE_CLICKED;
         if(interactive==null) {
            return;
         }
         interactive.handleMouseAction(InteractivePanel.this, e);
      }

      /**
       * Handles the mouse dragged event.
       * @param e
       */
      public void mouseDragged(MouseEvent e) {
         mouseEvent = e;
         mouseAction = MOUSE_DRAGGED;
         if(interactive!=null) {
            interactive.handleMouseAction(InteractivePanel.this, e);
         }
         if(showCoordinates) {
            String s = coordinateStrBuilder.getCoordinateString(InteractivePanel.this, e);
            blMessageBox.setText(s);
         }
      }

      /**
       * Handles the mouse moved event.
       * @param e
       */
      public void mouseMoved(MouseEvent e) {
         mouseEvent = e;
         mouseAction = MOUSE_MOVED;
         iaDragable = null;
         if(interactive!=null) { // check to see if there is an interactive object
            interactive.handleMouseAction(InteractivePanel.this, e);
            Interactive iad = getInteractive();
            if(iad==null) {
               setMouseCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
            } else {
               if(iad instanceof Selectable) {
                  setMouseCursor(((Selectable) iad).getPreferredCursor());
               } else {
                  setMouseCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
               }
            }
         }
      }
   }
}
