/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import org.opensourcephysics.controls.XML;
import org.opensourcephysics.controls.XMLControl;

/**
 * A measured circle that implements Interactive.
 *
 * The InteractiveCircle class also demonstrates how to implement a simple XML loader.
 *
 * @author Wolfgang Christian
 * @author Francisco Equembre
 * @version 1.0
 */
public class InteractiveCircle extends MeasuredCircle implements Interactive {

   boolean enableInteraction = true;

   /**
    * Constructs an InteractiveCircle with the given parameters.
    *
    * @param x
    * @param y
    */
   public InteractiveCircle (double x, double y) {
      super (x, y);
   }

   /**
    * Constructs an InteractiveCircle at the origin.
    */
   public InteractiveCircle() {
     this(0, 0);
   }

   /**
    * Enables mouse interactions.
    * @param _enableInteraction
    */
   public void setEnabled (boolean _enableInteraction) {
      enableInteraction = _enableInteraction;
   }

   public boolean isEnabled () {
      return enableInteraction;
   }

   public boolean isInside (DrawingPanel panel, int xpix, int ypix) {
      if (findInteractive(panel, xpix, ypix)==null ) {
         return false;
      } else return true;
   }

   public Interactive findInteractive (DrawingPanel panel, int xpix, int ypix) {
      if (!enableInteraction) {
         return null;
      }
      int xcpix = panel.xToPix (x); // convert x to pixel units
      int ycpix = panel.yToPix (y); // convert y to pixel units
      if ((Math.abs (xcpix - xpix) <= pixRadius) && (Math.abs (ycpix - ypix) <= pixRadius)) {
         return this;
      }
      return null;
   }

   public static XML.ObjectLoader getLoader() {
     return new InteractiveCircleLoader();
   }

/**
 * A class to save and load InteractiveCircle data in an XMLControl.
 */
private static class InteractiveCircleLoader extends CircleLoader {

    /**
     * Saves the InteractiveCircle's data in the xml control.
     * @param control XMLControl
     * @param obj Object
     */
    public void saveObject(XMLControl control, Object obj) {
    super.saveObject(control, obj);
    InteractiveCircle circle = (InteractiveCircle) obj;
    control.setValue("interaction enabled", circle.enableInteraction);
    control.setValue("measure enabled", circle.enableInteraction);
  }

  /**
   * Creates a default InteractiveCircle.
   * @param control XMLControl
   * @return Object
   */
  public Object createObject(XMLControl control) {
    return new InteractiveCircle();
  }

  /**
   * Loads data from the xml control into the InteractiveCircle object.
   * @param control XMLControl
   * @param obj Object
   * @return Object
   */
  public Object loadObject(XMLControl control, Object obj) {
    super.loadObject(control, obj);
    InteractiveCircle circle = (InteractiveCircle) obj;
    circle.enableInteraction=control.getBoolean("interaction enabled");
    circle.enableMeasure=control.getBoolean("measure enabled");
    return obj;
  }
}

}
