package org.opensourcephysics.display;

import java.awt.geom.GeneralPath;
import java.awt.geom.PathIterator;

import org.opensourcephysics.controls.XMLControl;
import org.opensourcephysics.controls.XMLLoader;

/**
 * A class that saves and loads a GeneralPath in an XMLControl.
 */
public class GeneralPathLoader extends XMLLoader {
  public void saveObject(XMLControl control, Object obj) {
    GeneralPath shape = (GeneralPath) obj;
    // iterator with line flatness better than 0.001
    PathIterator it = shape.getPathIterator(null, 0.001);
    control.setValue("winding rule", it.getWindingRule());
    control.setValue("segments", savePathSegments(it));
  }

  public Object createObject(XMLControl control) {
    return new GeneralPath();  // default shape is a GeneralPath
  }

  /**
   * Saves the path segments in a string.
   *
   * @param it PathIterator
   * @return String
   */
  String savePathSegments(PathIterator it) {
    StringBuffer sb = new StringBuffer();
    float[] coord = new float[6];
    double x1 = 0, y1 = 0;
    while(!it.isDone()) {
      switch(it.currentSegment(coord)) {
      case PathIterator.SEG_LINETO :
        x1 = coord[0];
        y1 = coord[1];
        sb.append("<LINETO "+x1+" "+y1+">");
        break;
      case PathIterator.SEG_MOVETO :
        x1 = coord[0];
        y1 = coord[1];
        sb.append("<MOVETO "+x1+" "+y1+">");
        break;
      case PathIterator.SEG_CLOSE :
        sb.append("<CLOSE>");
        break;
      default :
        System.out.println("Segment Type not supported. Type="+it.currentSegment(coord));
      }
      it.next();
    }
    return sb.toString();
  }

  /**
   * Loads the path with the segments from the given string.
   *
   * @param path GeneralPath
   * @param segments String
   */
  void loadPathSegments(GeneralPath path, String segments) {
    String[] segs = segments.split(">");
    for(int i = 0, n = segs.length;i<n;i++) {
      if(segs[i].startsWith("<LINETO ")) {
        String[] vals = segs[i].split(" ");
        path.lineTo(Float.parseFloat(vals[1]), Float.parseFloat(vals[2]));
      } else if(segs[i].startsWith("<MOVETO ")) {
        String[] vals = segs[i].split(" ");
        path.moveTo(Float.parseFloat(vals[1]), Float.parseFloat(vals[2]));
      } else if(segs[i].startsWith("<CLOSE")) {
        path.closePath();
      }
    }
  }

  public Object loadObject(XMLControl control, Object obj) {
    GeneralPath path = (GeneralPath) obj;
    path.reset();
    path.setWindingRule(control.getInt("winding rule"));
    loadPathSegments(path, control.getString("segments"));
    return path;
  }
}
