package org.opensourcephysics.display;
import java.io.*;
import javax.print.*;
import javax.print.attribute.*;
import javax.print.attribute.standard.*;
import java.awt.*;
import java.awt.print.*;
import javax.swing.*;
import java.util.Map;
import java.util.HashMap;

public class GUIUtils {

   public static Map charMap = new HashMap();
   static{
      // upper case
      charMap.put("\\Alpha", "\u0391");
      charMap.put("\\Beta", "\u0392");
      charMap.put("\\Gamma", "\u0393");
      charMap.put("\\Delta", "\u0394");
      charMap.put("\\Epsilon", "\u0395");
      charMap.put("\\Zeta", "\u0396");
      charMap.put("\\Eta", "\u0397");
      charMap.put("\\Theta", "\u0398");
      charMap.put("\\Pi", "\u03a0");
      charMap.put("\\Rho", "\u03a1");

      charMap.put("\\Sigma", "\u03a3");
      charMap.put("\\Tau", "\u03a4");
      charMap.put("\\Phi", "\u03a6");
      charMap.put("\\Chi", "\u03a7");
      charMap.put("\\Psi", "\u03a8");
      charMap.put("\\Omega", "\u03a9");
      charMap.put("\\Xi", "\u039e");
      // lower case
      charMap.put("\\alpha", "\u03b1");
      charMap.put("\\beta", "\u03b2");
      charMap.put("\\gamma", "\u03b3");
      charMap.put("\\delta", "\u03b4");
      charMap.put("\\epsilon", "\u03b5");
      charMap.put("\\zeta", "\u03b6");
      charMap.put("\\eta", "\u03b7");
      charMap.put("\\theta", "\u03b8");
      charMap.put("\\iota", "\u03b9");
      charMap.put("\\kappa", "\u03ba");
      charMap.put("\\lamda", "\u03bb");

      charMap.put("\\mu", "\u03bc");
      charMap.put("\\micro", "\u03bc");
      charMap.put("\\nu", "\u03bd");
      charMap.put("\\xi", "\u03be");
      charMap.put("\\pi", "\u03c0");
      charMap.put("\\rho", "\u03c1");
      charMap.put("\\sigma", "\u03c3");
      charMap.put("\\tau", "\u03c4");
      charMap.put("\\phi", "\u03c6");
      charMap.put("\\chi", "\u03c7");
      charMap.put("\\psi", "\u03c8");
      charMap.put("\\omega", "\u03c9");
      // special characters
      charMap.put("\\degree", "\u00b0");
   }


  private GUIUtils() {}  // prohibits instantiation

  public static String parseTeX(String inputStr){
     if(inputStr==null) return null;
     String[] chunks = inputStr.split("\\$");
     //boolean mathMode=(inputStr.charAt(0)=='$');
     boolean mathMode = false;
     for (int i = 0; i<chunks.length; i++){
        if (mathMode){
           String val = (String) charMap.get(chunks[i].trim());
           if (val!=null){
              chunks[i] = val;
           }
        }
        mathMode = !mathMode;
     }
     String outStr = "";
     for (int i = 0; i<chunks.length; i++){
        outStr += chunks[i];
     }
     return outStr;
  }


  /**
   * Shows all drawing and table frames.
   *
   * Usually invoked when a model is initialized but may be invoked at other times
   * to show frames that have been closed.
   */
  public static void showDrawingAndTableFrames() {
    Frame[] frames = Frame.getFrames();
    for(int i = 0;i<frames.length;i++) {
      if(!frames[i].isDisplayable()) {
        continue;
      }
      if((frames[i].getName()!=null)&&(frames[i].getName().indexOf("Tool")>-1)) {
        continue;
      }
      if(OSPFrame.class.isInstance(frames[i])) {
        if(DataTableFrame.class.isInstance(frames[i])) {
          ((DataTableFrame) frames[i]).refreshTable();
        }
        frames[i].setVisible(true);
        frames[i].repaint();  //repaint if frame is alrady showing
      }
    }
  }

  /**
   * Renders all OPSFrames whose animated property is true.
   *
   * Usually invoked by an anination thread after every animation step.
   */
  public static void renderOSPFrames() {
    Frame[] frames = Frame.getFrames();
    for(int i = 0;i<frames.length;i++) {
      if(!frames[i].isDisplayable()) {
        continue;
      }
      if(OSPFrame.class.isInstance(frames[i])&&((OSPFrame) frames[i]).isAnimated()) {
        ((OSPFrame) frames[i]).render();
      }
    }
  }

  /**
   * Repaints all OPSFrames whose animated property is true.
   *
   * Usually invoked by a control's single-step button.
   */
  public static void repaintOSPFrames(){
     Frame[] frames = Frame.getFrames();
     for (int i = 0; i<frames.length; i++){
        if (!frames[i].isDisplayable()){
           continue;
        }
        if (OSPFrame.class.isInstance(frames[i])&&
            ((OSPFrame) frames[i]).isAnimated()){
           ((OSPFrame) frames[i]).repaint();
        }
     }
  }


  /**
   * Clears the data in animated DrawingFrames and repaints the frame's content.
   *
   * All frames are cleared if <code> clearAll<\code> is true; otherwise only frames whose <code>autoClear<\code> flag is
   * true will be cleared.
   *
   * @param clearAll  clears all frames if true
   */
  public static void clearDrawingFrameData(boolean clearAll) {
    Frame[] frames = Frame.getFrames();
    for(int i = 0;i<frames.length;i++) {
      if(!frames[i].isDisplayable()) {
        continue;
      }
      if(DrawingFrame.class.isInstance(frames[i])&&((DrawingFrame) frames[i]).isAnimated()) {
        DrawingFrame frame = ((DrawingFrame) frames[i]);
        if(clearAll||frame.isAutoclear()) {
          frame.clearDataAndRepaint();
        }
      }
    }
  }

  /**
   * Enables and disables the menu bars in DRawingFrames.
   *
   * Usually invoked when a model is initialized but may be invoked at other times
   * to show frames that have been closed.
   */
  public static void enableMenubars(boolean enable) {
    Frame[] frames = Frame.getFrames();
    for(int i = 0;i<frames.length;i++) {
      if(!frames[i].isDisplayable()) {
        continue;
      }
      if((frames[i].getName()!=null)&&(frames[i].getName().indexOf("Tool")>-1)) {
        continue;
      }
      if(DrawingFrame.class.isInstance(frames[i])) {
        JMenuBar bar = ((DrawingFrame) frames[i]).getJMenuBar();
        if(bar!=null) {
          for(int j = 0, n = bar.getMenuCount();j<n;j++) {
            bar.getMenu(j).setEnabled(enable);
          }
        }
      }
    }
  }

  /**
   * Disposes all OSP frames except the given frame.
   *
   * Usually invoked when the control window is being closed.
   *
   * @param frame will not be disposed
   */
  public static void closeAndDisposeOSPFrames(Frame frame) {
    Frame[] frames = Frame.getFrames();
    for(int i = 0;i<frames.length;i++) {
      if(frames[i]==frame) {
        continue;
      }
      //if (frames[i] instanceof org.opensourcephysics.controls.Launcher.LauncherFrame)continue;
      if(OSPFrame.class.isInstance(frames[i])) {
        ((OSPFrame) frames[i]).setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        ((OSPFrame) frames[i]).setVisible(false);
        ((OSPFrame) frames[i]).dispose();
      }
    }
  }

  /**
   * Gets a random color.
   *
   * @return random color
   */
  public static Color randomColor() {
    return new Color((int) (Math.random()*255), (int) (Math.random()*255), (int) (Math.random()*255));
  }

  /**
   *   Pops up a "Save File" file chooser dialog and takes user through process of saving a file.
   *
   *   @param    parent  the parent component of the dialog,
   *                     can be <code>null</code>;
   *                    see <code>showDialog</code> in class JFileChooser for details
   *   @return   the file or null if an error occurred:
   */
  public static File showSaveDialog(Component parent) {
    JFileChooser fileChooser = new JFileChooser();
    fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    int result = fileChooser.showSaveDialog(parent);
    if(result!=JFileChooser.APPROVE_OPTION) {
      return null;
    }
    File file = fileChooser.getSelectedFile();
    if(file.exists()) {
      int selected = JOptionPane.showConfirmDialog(parent, "A file named "+file.getName()+" already exists. Are you sure you want to replace it?", "Warning", JOptionPane.YES_NO_CANCEL_OPTION);
      if(selected!=JOptionPane.YES_OPTION) {
        return null;
      }
    }
    return file;
  }

  public static File showOpenDialog(Component parent) {
    JFileChooser fileChooser = new JFileChooser();
    fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    int result = fileChooser.showOpenDialog(parent);
    if(result!=JFileChooser.APPROVE_OPTION) {
      return null;
    }
    File file = fileChooser.getSelectedFile();
    return file;
  }

  /**
   * Test the time to render a drawable component.
   * @param drawable
   */
  public static void timingTest(Drawable drawable) {
    DrawingPanel dp = new DrawingPanel();
    DrawingFrame df = new DrawingFrame(dp);
    df.setVisible(true);
    dp.addDrawable(drawable);
    dp.scale();
    dp.setPixelScale();
    Graphics g2 = dp.getGraphics();
    if(g2==null) {
      return;
    }
    long startTime = System.currentTimeMillis();
    drawable.draw(dp, g2);  // first drawing often takes longer because of initialization
    System.out.print("first drawing="+(System.currentTimeMillis()-startTime));
    startTime = System.currentTimeMillis();  // reset the time
    for(int i = 0;i<5;i++) {
      drawable.draw(dp, g2);
    }
    System.out.println("  avg time/drawing="+((System.currentTimeMillis()-startTime)/5));
    g2.dispose();
  }

  /**
   *  Saves the contents of the specified printable object as postscript to the
   *  specified output file. Note method requires Java 1.4
   *
   * @param  printable   the printable object to save as postscript
   * @param  outputFile  the output file
   */
  public static void saveAsPS(Printable printable, File outputFile) {
    //   Use the pre-defined flavor for a Printable from an InputStream
    DocFlavor flavor = DocFlavor.SERVICE_FORMATTED.PRINTABLE;
    // Specify the type of the output stream
    String psMimeType = DocFlavor.BYTE_ARRAY.POSTSCRIPT.getMimeType();
    StreamPrintServiceFactory[] factories = StreamPrintServiceFactory.lookupStreamPrintServiceFactories(flavor, psMimeType);
    if(factories.length==0) {
      System.err.println("No suitable factories");
      return;
    }
    try {
      // Create a file for the exported postscript
      FileOutputStream fos = new FileOutputStream(outputFile);
      // Create a Stream printer for Postscript
      StreamPrintService sps = factories[0].getPrintService(fos);
      // Create and call a Print Job
      DocPrintJob pj = sps.createPrintJob();
      PrintRequestAttributeSet aset = new HashPrintRequestAttributeSet();
      aset.add(MediaSizeName.ISO_A4);
      MediaSize m = new MediaSize(2, 2, Size2DSyntax.INCH);
      //aset.add(MediaSize.ISO);
      //aset.add(m);
      Doc doc = new SimpleDoc(printable, flavor, null);
      pj.print(doc, aset);
      fos.close();
    } catch(PrintException pe) {
      System.err.println(pe);
    } catch(IOException ie) {
      System.err.println(ie);
    }
  }

  /**
   *  Saves the contents of the specified printabe object as postscript. Pops
   *  open a save file dialog to allow the user to select the output file. Note
   *  method requires Java 1.4
   *
   * @param  parent     the parent component of the dialog, can be <code>null</code>
   * @param  printable  the printable object to save as postscript
   */
  public static void saveAsPS(Printable printable, Component parent) {
    File outputFile = GUIUtils.showSaveDialog(parent);
    if(outputFile==null) {
      return;
    }
    GUIUtils.saveAsPS(printable, outputFile);
  }
}
