/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.GeneralPath;

import org.opensourcephysics.numerics.Function;

/**
 * FunctionDrawer draws a function from xmin to xmax.
*
 * The function will be evaluated at every screen pixel unless the domain is set
 * using the initialize method.
 *
 * @author       Wolfgang Christian
 * @author       Joshua Gould
 * @version 1.0
 */
public class FunctionDrawer
    implements Drawable, Measurable, Function {
    private double[] xrange = new double[2];
    private double[] yrange = new double[2];
    private int numpts=0;
    private GeneralPath generalPath = new GeneralPath();
    private Function function;
    private boolean filled=false;
    private boolean measured=false;  // set to true if function has been initialized.

    public Color color=Color.black;

    /**
     * Contstucts a FunctionDrawer with optimum resolution.
     *
     * @param     Function f the function that will be drawn.
     */
    public FunctionDrawer (Function f) {
        function = f;
    }

    /**
     * Creates the function drawer and initialzies the domain with the given values.
     * @param f Function
     * @param xmin double
     * @param xmax double
     * @param numpts int
     * @param filled boolean  fills the area under the curve with the drawing color when true
     */
    public FunctionDrawer (Function f, double xmin, double xmax, int numpts, boolean filled) {
      this(f);
      initialize( xmin,  xmax,  numpts,  filled);
    }


    /**
     * Evalutes the function.
     * @param x
     * @return value of the function
     */
    public double evaluate(double x){
      return function.evaluate(x);
    }

    /**
     * Initialize the function range and the number of display points.
     * @param xmin  the beginning value of the range.
     * @param xmax  the ending value for the range
     * @param numPts the number of points to display
     * @param filled fills the area under the curve with the drawing color when true
     */
    public void initialize (double xmin, double xmax, int numpts, boolean filled) {
        if (numpts < 1) return;
        this.filled=filled;
        this.xrange[0] = xmin;
        this.xrange[1] = xmax;
        this.numpts=numpts;
        generalPath.reset();
        if (numpts<1) return;
        yrange[0]=function.evaluate(xmin);
        yrange[1]=yrange[0]; // starting values for ymin and ymax
        if(filled){
          generalPath.moveTo((float)xrange[0], 0);
          generalPath.lineTo((float)xrange[0], (float)yrange[0]);
        }else generalPath.moveTo((float)xrange[0], (float)yrange[0]);
        double x=xrange[0];
        double dx=(xmax-xmin)/(numpts);
        for (int i=0; i<numpts; i++){
          x=x+dx;
          double y=function.evaluate(x);
          generalPath.lineTo((float)x, (float)y);
          if(y<yrange[0]) yrange[0]=y;  // the minimum value
          if(y>yrange[1]) yrange[1]=y;  // the maximum value
        }
        if(filled){
          generalPath.lineTo((float)x, 0);
          generalPath.closePath();
        }
        measured=true;
    }

    /**
     * Get the range of x values over which the function has been evaluated.
     *
     * @return double[2]      the xmin and xmax values
     */
    public double[] getXRange(){ return xrange;}

    /**
     * Get the minimum and maximum y values for the function.
     *
     * @return double[2]      the ymin and ymax values
     */
    public double[] getYRange(){ return yrange;}

    void checkRange(DrawingPanel panel){
      if(xrange[0] == panel.getXMin() && // check to see if the range has changed
         xrange[1] == panel.getXMax() &&
         numpts==panel.getWidth()){
            return;
      }
      xrange[0] = panel.getXMin();
      xrange[1] = panel.getXMax();
      numpts=panel.getWidth();

      generalPath.reset();
      if (numpts < 1)return;
      yrange[0] = function.evaluate(xrange[0]);
      yrange[1] = yrange[0]; // starting values for ymin and ymax
      if (filled) {
        generalPath.moveTo( (float) xrange[0], 0);
        generalPath.lineTo( (float) xrange[0], (float) yrange[0]);
      }
      else generalPath.moveTo( (float) xrange[0], (float) yrange[0]);
      double x = xrange[0];
      double dx = (xrange[1] - xrange[0]) / (numpts);
      for (int i = 0; i < numpts; i++) {
        x = x + dx;
        double y = function.evaluate(x);
        generalPath.lineTo( (float) x, (float) y);
        if (y < yrange[0]) yrange[0] = y; // the minimum value
        if (y > yrange[1]) yrange[1] = y; // the maximum value
      }
      if (filled) {
        generalPath.lineTo( (float) x, 0);
        generalPath.closePath();
      }
    }

    /**
     * Draw the function on a drawing panel.
     * @param panel  the drawing panel
     * @param g      the graphics context
     */
    public void draw (DrawingPanel panel, Graphics g) {
        if(!measured)checkRange(panel);
        Graphics2D g2 = (Graphics2D)g;
        g2.setColor(color);
        // transform from world to pixel coordinates
        Shape s = generalPath.createTransformedShape(panel.getPixelTransform());
        if(filled){
          g2.fill(s);
          g2.draw(s);
        }else g2.draw(s);
    }

    /**
     * Fills the area under the curve when true.
     * @param _filled boolean
     */
    public void setFilled(boolean _filled){filled=_filled;}

    /**
     * Sets the drawing color.
     * @param c Color
     */
    public void setColor(Color c){color=c;}

    //Implementation of measured interface.
    public boolean isMeasured(){return measured;}
    public double getXMin(){ return xrange[0];}
    public double getXMax(){ return xrange[1];}
    public double getYMin(){ return yrange[0];}
    public double getYMax(){ return yrange[1];}
}



