/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * Draws a group of shapes.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */
public class DrawableGroup implements Drawable {
  protected double x=0, y=0, theta=0;
  protected ArrayList  drawableList  = new ArrayList();         // list of Drawable objects

  /**
   * Adds a drawable object to the drawable list.
   * @param drawable
   */
  public void addDrawable(Drawable drawable) {
    if((drawable != null) &&!drawableList.contains(drawable)) {
      drawableList.add(drawable);
    }
  }

  /**
   * Draws the shapes in the drawable list.
   *
   * @param panel the drawing panel
   * @param g  the graphics context
   */
  public void draw(DrawingPanel panel, Graphics g) {
    int xpix = panel.xToPix (0);
    int ypix = panel.yToPix (0);
    Graphics2D g2        = (Graphics2D) g;
    Iterator   it        = drawableList.iterator();
    AffineTransform oldAT=g2.getTransform();
    AffineTransform at=g2.getTransform();
    at.concatenate(AffineTransform.getRotateInstance(-theta,xpix,ypix));
    double xt=x*panel.getXPixPerUnit()*Math.cos(theta)+y*panel.getYPixPerUnit()*Math.sin(theta);
    double yt=x*panel.getXPixPerUnit()*Math.sin(theta)-y*panel.getYPixPerUnit()*Math.cos(theta);
    at.concatenate(AffineTransform.getTranslateInstance(xt,yt));
    g2.setTransform(at);
    while(it.hasNext()) {
      Drawable drawable = (Drawable) it.next();
      drawable.draw(panel, g2);
    }
    g2.setTransform(oldAT);
  }

  /**
 * Sets the x and y coordinates.
 *
 * @param y
 */
public void setXY (double _x, double _y) {
  x=_x;
  y=_y;
}

/**
 * Sets the x coordinate.
 *
 * @param x
 */
public void setX (double _x) {
  x=_x;
}

/**
 * Gets the x location.
 * @return x
 */
public double getX(){
  return x;
}

/**
 * Gets the y location.
 * @return y
 */
public double getY(){
  return y;
}



/**
 * Sets the y coordinate.
 *
 * @param y
 */
public void setY (double _y) {
  y=_y;
}

/**
 * Gets the rotation angle in radians.
 * @return theta
 */
public double getTheta(){
  return theta;
}

/**
 * Sets the rotation angle in radians.
 *
 * @param x
 */
public void setTheta (double _theta) {
  theta=_theta;
}

}
