/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework desizefines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * A DrawableBuffer contains an image of drawable objects.  This image is displayed
 * on a drawing panel whenever when the draw method is invoked.
 *
 * A drawble buffer should be used to render complex drawable objects that change infrequently.
 * Use the updateImage method to generate a new image when the properties of
 * the drawable objects change.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class DrawableBuffer implements Drawable, Measurable {

  Image     image;
  boolean   invalid      = true;
  ArrayList drawableList = new ArrayList ();  // list of Drawable objects
  Color     background   = Color.white;
  boolean   measured     = false;
  double    xmin, xmax, ymin, ymax;

  /**
   * Constructor DrawableBuffer
   *
   */
  public DrawableBuffer () {}

  /**
   * Constructor DrawableBuffer
   *
   * @param drawable
   */
  public DrawableBuffer (Drawable drawable) {
    addDrawable (drawable);
  }

  /**
   * Adds a drawable object to the drawing buffer.
   * @param drawable
   */
  public synchronized void addDrawable (Drawable drawable) {
    if(!drawableList.contains(drawable)) drawableList.add (drawable);
    invalidateImage ();
  }

  /**
   * Method setBackground
   *
   * @param color
   */
  public void setBackground (Color color) {
    background = color;
  }

  /**
   * Remove all drawable objects from the drawing buffer.
   */
  public synchronized void clear () {
    drawableList.clear ();
    invalidateImage ();
  }

  /**
   * Invalidates the image so that it is redrawn during the next repaint operation.
   *
   */
  public void invalidateImage () {
    measured = false;
    ArrayList tempList = (ArrayList) drawableList.clone ();
    Iterator  it       = tempList.iterator ();
    xmin=Double.MAX_VALUE;
    xmax=-Double.MAX_VALUE;
    ymin=Double.MAX_VALUE;
    ymax=-Double.MAX_VALUE;
    while (it.hasNext ()) {
      Drawable drawable = (Drawable) it.next ();
      if ((drawable instanceof Measurable) && ((Measurable) drawable).isMeasured ()) {
        xmin     = Math.min (xmin, ((Measurable) drawable).getXMin ());
        xmax     = Math.max (xmax, ((Measurable) drawable).getXMax ());
        ymin     = Math.min (ymin, ((Measurable) drawable).getYMin ());
        ymax     = Math.max (ymax, ((Measurable) drawable).getYMax ());
        measured = true;
      }
    }
    invalid  = true;
  }

  /**
   * Updates the image using the given drawing panel to set the dimension.
   *
   * @param drawingPanel
   */
  public void updateImage (DrawingPanel drawingPanel) {
    invalid = false;
    Image newImage = image;
    int   iw       = drawingPanel.getWidth ();
    int   ih       = drawingPanel.getHeight ();
    if ((image == null) || (image.getWidth (drawingPanel) != iw) || (image.getHeight (drawingPanel) != ih)) {
      newImage = drawingPanel.createImage (iw, ih);
    }
    if (newImage == null) {
      return;
    }
    Graphics g = newImage.getGraphics ();
    if (g != null) {
      if (background == null) {
        g.clearRect (0, 0, iw, ih);
      } else {
        g.setColor (background);
        g.fillRect (0, 0, iw, ih);
      }
      paintMyDrawableList (drawingPanel, g);
      g.dispose ();
    }
    image = newImage;
  }

  /**
   * Method draw
   *
   * @param drawingPanel
   * @param g
   */
  public void draw (DrawingPanel drawingPanel, Graphics g) {
    if (invalid || (image == null) || (image.getWidth (drawingPanel) != drawingPanel.getWidth ())
        || (image.getHeight (drawingPanel) != drawingPanel.getHeight ())) {
      updateImage (drawingPanel);
    }
    if (image != null) {
      g.drawImage (image, 0, 0, drawingPanel);
    }
  }

  /**
   * Method getXMin
   *
   *
   * @return
   */
  public double getXMin () {
    return xmin;
  }

  /**
   * Method getXMax
   *
   *
   * @return
   */
  public double getXMax () {
    return xmax;
  }

  /**
   * Method getYMin
   *
   *
   * @return
   */
  public double getYMin () {
    return ymin;
  }

  /**
   * Method getYMax
   *
   *
   * @return
   */
  public double getYMax () {
    return ymax;
  }

  /**
   * Tests to see if the buffer has an object with a valid measure.
   *
   * @return true if any object in the drawable list is measured
   */
  public boolean isMeasured () {
    return measured;
  }

  /**
   * Paints the drawable objects onto the image.
   * @param g
   */
  private void paintMyDrawableList (DrawingPanel drawingPanel, Graphics g) {
    ArrayList tempList = (ArrayList) drawableList.clone ();
    Iterator  it       = tempList.iterator ();
    while (it.hasNext ()) {
      Drawable drawable = (Drawable) it.next ();
      drawable.draw (drawingPanel, g);
    }
  }
}
