/*
 * The org.opensourcephysics.display package contains the drawing framework
 * for the book Simulations in Physics.  This framework defines objects that
 * implement the Drawable interface and a DrawingPanel for rendering these objects.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.display;

import java.awt.Color;
import java.awt.Graphics;

import org.opensourcephysics.controls.XML;

/**
 * A Drawable circle that uses awt drawing.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */
public class Circle implements Drawable {
   public Color  color     = Color.red;  // the drawing color
   public int    pixRadius = 6;
   protected double x         = 0;
   protected double y         = 0;

   /**
    * Constructs a fixed radius circle at the origin.
    */
   public Circle() {
     this(0,0);
   }

   /**
    * Constructs a fixed radius circle at the given coordinates.
    *
    * The default radius is 6 pixels.
    *
    * @param x
    * @param y
    */
   public Circle (double _x, double _y) {
      x = _x;
      y = _y;
   }

   /**
    * Constructs a fixed radius circle at the given coordinates with the given radius.
    *
    * The radius is given in pixels.
    *
    * @param x
    * @param y
    * @param r
    */
   public Circle (double _x, double _y, int _r) {
      x = _x;
      y = _y;
      pixRadius=_r;
   }

   /**
    * Draws the circle.
    *
    * @param panel
    * @param g
    */
   public void draw (DrawingPanel panel, Graphics g) {
      int xpix = panel.xToPix (x) - pixRadius;
      int ypix = panel.yToPix (y) - pixRadius;
      g.setColor (color);
      g.fillOval (xpix, ypix, 2 * pixRadius, 2 * pixRadius);  // draw the circle onto the screen
   }

   /**
    * Gets the x coordinate.
    *
    * @return double x
    */
   public double getX () {
      return x;
   }

   /**
    * Sets the x coordinate.
    *
    * @param x
    */
   public void setX (double x) {
      this.x=x;
   }

   /**
    * Gets the y coordinate.
    *
    * @return double y
    */
   public double getY () {
      return y;
   }

   /**
    * Sets the y coordinate.
    *
    * @param y
    */
   public void setY (double y) {
      this.y=y;
   }

   /**
    * Sets the x and y coordinates.
    *
    * @param x
    * @param y
    */
   public void setXY(double x, double y) {
     this.x = x;
     this.y = y;
   }

   /**
    * Returns a string representation of the circle.
    *
    * @return String
    */
   public String toString(){
      String name = getClass().getName();
      name = name.substring(1+name.lastIndexOf("."))+'[';
      name+= "x="+x;
      name+= ",y="+y;
      name+= ",r_pix="+pixRadius+']';
      return name;
   }

   /**
    * Gets a loader that allows a Circle to be represented as XML data.
    * Objects without XML loaders cannot be saved and retrieved from an XML file.
    *
    * @return ObjectLoader
    */
   public static XML.ObjectLoader getLoader() {
     return new CircleLoader();
   }
}
