/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

/**
 * This is a split pane view of an XML tree and its contents.
 *
 * @author Douglas Brown
 */
public class XMLTreePanel extends JPanel {

  // instance fields
  protected JLabel label;
  protected JTextField input;
  protected JTextPane xmlPane;
  protected JTree tree;
  protected XMLControl control;
  protected XMLProperty property;
  protected boolean changeEnabled;
  int maxStringLength = 24;

  /**
   * Contructs a tree panel with an XMLControl
   *
   * @param control the XMLControl
   */
  public XMLTreePanel(XMLControl control) {
    this(control, true);
  }

  /**
   * Contructs a tree panel with an XMLControl
   *
   * @param control the XMLControl
   * @param enabled true to enable xml changes via the input field
   */
  public XMLTreePanel(XMLControl control, boolean enabled) {
    super(new BorderLayout());
    this.control = control;
    changeEnabled = enabled;
    createGUI();
  }

  /**
   * Displays the property data for the specified node.
   *
   * @param node the XMLTreeNode
   * @param enabled <code>true</code> if the input field is enabled
   */
  protected void displayProperty(XMLTreeNode node, boolean enabled) {
    // input field hidden by default
    input.setVisible(false);
    XMLProperty prop = node.getProperty();
    // display property type and name on label
    label.setText(prop.getPropertyType() + " " + prop.getPropertyName());
    if (!prop.getPropertyContent().isEmpty()) {
      // get first content item
      Object value = prop.getPropertyContent().get(0);
      // display primitive properties in input field
      if (value instanceof String) {
        property = prop;
        input.setText( (String) value);
        input.setEnabled(enabled);
        input.setVisible(true);
      }
    }
    // display xml in xmlPane
    String xml = prop.toString();
    xmlPane.setText(getDisplay(xml));
    xmlPane.setCaretPosition(0);
  }

  /**
   * Gets the xml to be displayed.
   *
   * @param xml the raw xml
   * @return the displayed xml
   */
  protected String getDisplay(String xml) {
    // find and truncate every array string in the xml
    String newXML = ""; // newly assembled xml
    String preArray = "name=\"array\" type=\"string\">";
    String postArray = "</property>";
    String array;
    int i = xml.indexOf(preArray);
    while (i > 0) {
      i += preArray.length();
      newXML += xml.substring(0, i);
      xml = xml.substring(i);
      i = xml.indexOf(postArray);
      array = xml.substring(0, i);
      xml = xml.substring(i, xml.length());
      if (array.length() > maxStringLength) {
        array = array.substring(0, maxStringLength - 3) + "...";
      }
      newXML += array;
      i = xml.indexOf(preArray);
    }
    newXML += xml;
    return newXML;
  }

  /**
   * Creates the GUI and listeners.
   */
  protected void createGUI() {
    // create root and tree
    XMLTreeNode root = new XMLTreeNode(control);
    tree = new JTree(root);
    tree.getSelectionModel().setSelectionMode
            (TreeSelectionModel.SINGLE_TREE_SELECTION);
    // listen for tree selections and display the property data
    tree.addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
        XMLTreeNode node = (XMLTreeNode) tree.getLastSelectedPathComponent();
        if (node != null) displayProperty(node, changeEnabled);
      }
    });
    // listen for mouse double-clicks to display array tables
    tree.addMouseListener(new MouseAdapter() {
       public void mouseClicked(MouseEvent e) {
         if (e.getClickCount() == 2) {
           XMLTreeNode node = (XMLTreeNode)tree.getLastSelectedPathComponent();
           if (node != null &&
               node.getProperty().getPropertyType().equals("array")) {
             XMLProperty prop = node.getProperty();
             // get base component type and depth
             Class type = prop.getPropertyClass();
             if (type == null) return;
             while (type.getComponentType() != null) {
               type = type.getComponentType();
             }
             if (type.getName().equals("double") ||
                 type.getName().equals("int")) { // node is double or int array
               String name = node.getProperty().getPropertyName();
               XMLProperty parent = node.getProperty().getParentProperty();
               while (!(parent instanceof XMLControl)) {
                 name = parent.getPropertyName();
                 prop = parent;
                 parent = parent.getParentProperty();
               }
               // get array depth
               type = prop.getPropertyClass();
               int i = 0;
               while (type.getComponentType() != null) {
                 type = type.getComponentType();
                 i++;
               }
               XMLControl control = (XMLControl) parent;
               Object obj = control.getObject(name);
               if (obj == null) return;
               org.opensourcephysics.tools.ArrayInspector table = null;
               if (type.getName().equals("double")) {
                 if (i == 1) {
                   double[] array = (double[]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
                 else if (i == 2) {
                   double[][] array = (double[][]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
                 else if (i == 3) {
                   double[][][] array = (double[][][]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
               }
               else if (type.getName().equals("int")) {
                 if (i == 1) {
                   int[] array = (int[]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
                 else if (i == 2) {
                   int[][] array = (int[][]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
                 else if (i == 3) {
                   int[][][] array = (int[][][]) obj;
                   table = new org.opensourcephysics.tools.ArrayInspector(array,
                       name);
                 }
               }
               if (table != null)
                 table.setVisible(true);
             }
           }
         }
       }
    });
    // put tree in a scroller for left side of split pane
    JScrollPane treeScroller = new JScrollPane(tree);
    treeScroller.setPreferredSize(new Dimension(140, 200));
    // create toolbar for label and input text field
    JToolBar toolbar = new JToolBar();
    toolbar.setFloatable(false);
    // create label
    label = new JLabel();
    toolbar.add(label);
    // create input text field
    input = new JTextField(20);
    input.setVisible(false);
    input.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        property.setValue(input.getText());
        Object obj = control.loadObject(null);
        if (obj instanceof Component) {
          ((Component)obj).repaint();
        }
        input.setText((String)property.getPropertyContent().get(0));
        input.selectAll();
        XMLTreeNode node = (XMLTreeNode) tree.getLastSelectedPathComponent();
        if (node != null) displayProperty(node, changeEnabled);
      }
    });
    input.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
        JComponent comp = (JComponent)e.getSource();
        if (e.getKeyCode() == KeyEvent.VK_ENTER) {
          comp.setBackground(Color.white);
        }
        else {
          comp.setBackground(Color.yellow);
        }
      }
    });
    input.addFocusListener(new FocusAdapter() {
      public void focusLost(FocusEvent e) {
        JComponent comp = (JComponent)e.getSource();
        comp.setBackground(Color.white);
      }
    });
    toolbar.add(input);
    // create xml pane and scroller
    xmlPane = new JTextPane();
    xmlPane.setPreferredSize(new Dimension(360, 200));
    xmlPane.setEditable(false);
    JScrollPane xmlScroller = new JScrollPane(xmlPane);
    // create data panel for right side of split pane
    JPanel dataPanel = new JPanel(new BorderLayout());
    dataPanel.add(toolbar, BorderLayout.NORTH);
    dataPanel.add(xmlScroller, BorderLayout.CENTER);
    // create split pane
    JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
                                          treeScroller, dataPanel);
    add(splitPane, BorderLayout.CENTER);
    displayProperty(root, changeEnabled);
  }
}
