/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * This modal dialog lets the user choose items from a tree view of an XMLControl.
 */
public class XMLTreeChooser extends JDialog {

  // instance fields
  private JPanel scrollPane;
  private XMLTree tree;
  private JLabel textLabel;
  private boolean applyChanges = false;

  /**
   * Constructs a dialog with the specified title and text.
   *
   * @param title the title of the dialog
   * @param text the label text
   */
  public XMLTreeChooser(String title, String text) {
    this(title, text, null);
  }

  /**
   * Constructs a dialog with the specified title, text and owner.
   *
   * @param title the title of the dialog
   * @param text the label text
   * @param comp the component that owns the dialog (may be null)
   */
  public XMLTreeChooser(String title, String text, Component comp) {
    super(JOptionPane.getFrameForComponent(comp), true);
    setTitle(title);
    textLabel = new JLabel(" " + text);
    textLabel.setHorizontalTextPosition(JLabel.LEFT);
    // create the buttons
    JButton cancelButton = new JButton("Cancel");
    JButton okButton = new JButton("OK");
    JButton selectAllButton = new JButton("Select_All");
    cancelButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        setVisible(false);
      }
    });
    okButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        applyChanges = true;
        setVisible(false);
      }
    });
    selectAllButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        tree.selectHighlightedProperties();
      }
    });
    getRootPane().setDefaultButton(okButton);
    // lay out the header pane
    JPanel headerPane = new JPanel();
    headerPane.setLayout(new BoxLayout(headerPane, BoxLayout.X_AXIS));
    headerPane.add(textLabel);
    headerPane.add(Box.createHorizontalGlue());
    headerPane.add(selectAllButton);
    headerPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
    // lay out the scroll pane
    scrollPane = new JPanel(new BorderLayout());
    scrollPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
    // lay out the button pane
    JPanel buttonPane = new JPanel();
    buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
    buttonPane.setBorder(BorderFactory.createEmptyBorder(0, 10, 10, 10));
    buttonPane.add(Box.createHorizontalGlue());
    buttonPane.add(cancelButton);
    buttonPane.add(Box.createRigidArea(new Dimension(10, 0)));
    buttonPane.add(okButton);
    // add everything to the content pane
    JPanel contentPane = new JPanel(new BorderLayout());
    contentPane.setPreferredSize(new Dimension(340, 340));
    setContentPane(contentPane);
    contentPane.add(headerPane, BorderLayout.NORTH);
    contentPane.add(scrollPane, BorderLayout.CENTER);
    contentPane.add(buttonPane, BorderLayout.SOUTH);
    pack();
    // center dialog on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - this.getBounds().width) / 2;
    int y = (dim.height - this.getBounds().height) / 2;
    setLocation(x, y);
  }

  /**
   * Highlights objects of a specified class in a tree view of an XMLControl
   * and returns those selected by the user, if any.
   *
   * @param control the XMLControl
   * @param type the class to be highlighted
   * @return a list of selected objects
   */
  public java.util.List choose(XMLControl control, Class type) {
    ArrayList list = new ArrayList();
    tree = new XMLTree(control);
    tree.setHighlightedClass(type);
//    tree.showHighlightedProperties();
    tree.selectHighlightedProperties();
    textLabel.setIcon(XMLTree.hiliteIcon);
    scrollPane.removeAll();
    scrollPane.add(tree.getScrollPane(), BorderLayout.CENTER);
    validate();
    applyChanges = false;
    setVisible(true);
    if (applyChanges) {
      java.util.List props = tree.getSelectedProperties();
      Iterator it = props.iterator();
      while (it.hasNext()) {
        XMLProperty prop = (XMLProperty)it.next();
        Class propClass = prop.getPropertyClass();
        if (propClass != null && type.isAssignableFrom(propClass)) {
          list.add(prop);
        }
      }
    }
    return list;
  }

}
