/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.io.Reader;
import java.io.Writer;

/**
 * This defines methods for storing data in an xml control element.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public interface XMLControl extends Control, XMLProperty {


  /**
   * Gets the type of the specified property.
   *
   * @param name the property name
   * @return the type
   */
  public String getPropertyType(String name);


  /**
   * Gets the name of the object class for which this element stores data.
   *
   * @return the object class name
   */
  public String getObjectClassName();

  /**
   * Gets the class for which this stores data.
   *
   * @return the class
   */
  public Class getObjectClass();

  /**
   * Saves an object's data in this XMLControl.
   *
   * @param obj the object to save.
   */
  public void saveObject(Object obj);

  /**
   * Loads an object with data from this XMLControl.
   *
   * @param obj the object to load
   * @return the loaded object
   */
  public Object loadObject(Object obj);

  /**
   * Reads the control from an xml document with the specified name.
   *
   * @param name the name
   * @return the full name of the opened document or null if failed
   */
  public String read(String name);

  /**
   * Reads the control from a Reader.
   *
   * @param reader the Reader
   */
  public void read(Reader reader);

  /**
   * Reads the control from an xml string.
   *
   * @param xml the xml string
   */
  public void readXML(String xml);

  /**
   * Returns true if the most recent read operation failed.
   *
   * @return <code>true</code> if the most recent read operation failed
   */
  public boolean failedToRead();

  /**
   * Writes the control as an xml document with the specified name.
   *
   * @param fileName the file name
   * @return the full name of the opened document or null if failed
   */
  public String write(String fileName);

  /**
   * Writes the control to a Writer.
   *
   * @param writer the Writer
   */
  public void write(Writer writer);

  /**
   * Returns this control as an xml string.
   *
   * @return the xml string
   */
  public String toXML();

}
