/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;
import java.awt.Color;
import java.io.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * This defines the ObjectLoader interface and static methods for managing and
 * accessing ObjectLoader implementations.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class XML {

  // static constants
  public static String NEW_LINE = "/n";
  public static final String CDATA_PRE = "<![CDATA[";
  public static final String CDATA_POST = "]]>";
  public static final int INDENT = 4;

  // static fields
  private static Map loaders = new HashMap();
  private static ObjectLoader defaultLoader;
  private static String dtdName;
  private static String dtd; // the dtd as a string
  private static String defaultName = "osp10.dtd";

  // added by W. Christian
  static {
    try { // system properties may not be readable in some environments
      NEW_LINE = System.getProperty("line.separator", "/n");
    }
    catch (SecurityException ex) {}
  }

  /**
   * Private constructor to prevent instantiation.
   */
  private XML() {}

  /**
   * Sets the ObjectLoader for a specified class.
   *
   * @param classtype the class
   * @param loader the ObjectLoader
   */
  public static void setLoader(Class classtype, XML.ObjectLoader loader) {
    loaders.put(classtype, loader);
  }

  /**
   * Gets the ObjectLoader for the specified class.
   *
   * @param classtype the class
   * @return the ObjectLoader
   */
  public static XML.ObjectLoader getLoader(Class classtype) {
    // look for registered loader first
    ObjectLoader loader = (ObjectLoader)loaders.get(classtype);
    // if no registered loader, look for static getLoader() method in class
    if (loader == null) {
      try {
        Method method = classtype.getMethod("getLoader", (Class[])null);
        if (method != null && Modifier.isStatic(method.getModifiers())) {
          loader = (ObjectLoader) method.invoke(null, (Object[])null);
          if (loader != null) {
            // register loader for future calls
            setLoader(classtype, loader);
          }
        }
      }
      catch (IllegalAccessException ex) {}
      catch (IllegalArgumentException ex) {}
      catch (InvocationTargetException ex) {}
      catch (NoSuchMethodException ex) {}
      catch (SecurityException ex) {}
    }
    // if still no loader found, use the default loader
    if (loader == null) {
      if (defaultLoader == null) {
        defaultLoader = new XMLLoader();
      }
      loader = defaultLoader;
    }
    return loader;
  }

  /**
   * Sets the default ObjectLoader. May be set to null.
   *
   * @param loader the ObjectLoader
   */
  public static void setDefaultLoader(XML.ObjectLoader loader) {
    defaultLoader = loader;
  }

  /**
   * Gets the datatype of the object.
   *
   * @param obj the object
   * @return the type
   */
  public static String getDataType(Object obj) {
    if (obj == null) return null;
    if (obj instanceof String) return "string";
    else if (obj instanceof Collection) return "collection";
    else if (obj.getClass().isArray()) {
      // make sure ultimate component class is acceptable
      Class componentType = obj.getClass().getComponentType();
      while (componentType.isArray()) {
        componentType = componentType.getComponentType();
      }
      String type = componentType.getName();
      if (type.indexOf(".") == -1 && "intdoubleboolean".indexOf(type) == -1)
        return null;
      return "array";
    }
    else return "object";
  }

  /**
   * Gets an array containing all supported data types.
   *
   * @return an array of types
   */
  public static String[] getDataTypes() {
    return new String[] {"object",
                         "array",
                         "collection",
                         "string",
                         "int",
                         "double",
                         "boolean"};
  }

  /**
   * Determines whether the specified string requires CDATA tags.
   *
   * @param text the string
   * @return <code>true</code> if CDATA tags are required
   */
  public static boolean requiresCDATA(String text) {
    if (text.indexOf("\"") != -1 ||
        text.indexOf("<") != -1 ||
        text.indexOf(">") != -1 ||
        text.indexOf("&") != -1 ||
        text.indexOf("'") != -1) {
      return true;
    }
    return false;
  }

  /**
   * Gets the DTD for the specified doctype file name.
   *
   * @param doctype the doctype file name (e.g., "osp10.dtd")
   * @return the DTD as a string
   */
  public static String getDTD(String doctype) {
    if (dtdName != doctype) {
      // set to defaults in case doctype is not found
      dtdName = defaultName;
      try {
        String dtdPath="/org/opensourcephysics/resources/controls/doctypes/";
        java.net.URL url = XML.class.getResource(dtdPath + doctype);
        if (url == null) return dtd;
        Object content = url.getContent();
        if (content instanceof InputStream) {
          BufferedReader reader = new BufferedReader(
              new InputStreamReader( (InputStream) content));
          StringBuffer buffer = new StringBuffer(0);
          String line;
          while ( (line = reader.readLine()) != null) {
            buffer.append(line + NEW_LINE);
          }
          dtd = buffer.toString();
          dtdName = doctype;
        }
      }
      catch (IOException ex) {}
    }
    return dtd;
  }

  /**
   * Gets a path relative to the specified base directory.
   *
   * @param path the path
   * @param base the base directory path
   * @return the path relative to the base,
   * or the absolute path if no relative path is found
   */
  public static String getPathRelativeTo(String path, String base) {
    OSPLog.info("getting path of " + path + " relative to " + base);
    int jar = path.indexOf("jar!");
    if (jar > -1) {
      path = path.substring(jar + 5);
      OSPLog.info("returning " + path);
      return path;
    }
    if (path.indexOf("/") == -1 && path.indexOf("\\") == -1) return path;
    String relativePath = "";
    boolean validPath = false;
    // search for base in containing folders up to 4 above
    for (int j = 0; j < 4; j++) {
      if (j > 0) {
        // move up one level
        int k = base.lastIndexOf("\\");
        if (k == -1)
          k = base.lastIndexOf("/");
        if (k != -1) {
          base = base.substring(0, k); // doesn't include the slash
          relativePath += "../";
        }
        else if (!base.equals("")) {
          base = "";
          relativePath += "../";
        }
        else break; // no more levels
      }
      if (path.startsWith(base)) {
        path = path.substring(base.length());
        // eliminate leading slash, if any
        int k = path.indexOf("\\");
        if (k == -1)
          k = path.indexOf("/");
        if (k == 0) {
          path = path.substring(1);
        }
        // replace backslashes with forward slashes
        int i = path.indexOf("\\");
        while (i != -1) {
          path = path.substring(0, i) + "/" + path.substring(i + 1);
          i = path.indexOf("\\");
        }
        relativePath += path;
        validPath = true;
        break;
      }
    }
    if (validPath) {
      OSPLog.info("returning " + relativePath);
      return relativePath;
    }
    OSPLog.info("returning " + path);
    return path;
  }

  /**
   * Gets a path relative to the default user directory.
   *
   * @param absolutePath the absolute path
   * @return the relative path
   */
  public static String getRelativePath(String absolutePath) {
    return getPathRelativeTo(absolutePath, System.getProperty("user.dir"));
    // todo: relative to doc base, code base, xml file?
  }

  /**
   * Gets the path of the directory containing the specified file.
   *
   * @param fileName the full file name, including path
   * @return the directory path
   */
  public static String getDirectoryPath(String fileName) {
    if (fileName == null) return "";
    int slash = fileName.lastIndexOf("/");
    if (slash == -1) slash = fileName.lastIndexOf("\\");
    if (slash != -1) return fileName.substring(0, slash);
    else return "";
  }

  /**
   * Resolves the name of a file specified relative to a path.
   *
   * @param name the file name
   * @param path the path
   * @return the resolved file name
   */
  public static String getName(String name, String path) {
    while (name.startsWith("../") && !path.equals("")) {
      if (path.indexOf("/") == -1) {
        path = "/" + path;
      }
      name = name.substring(3);
      path = path.substring(0, path.indexOf("/"));
    }
    if (path.equals("")) return name;
    if (path.endsWith("/")) return path + name;
    return path + "/" + name;
  }

//_________________________ ObjectLoader interface _____________________________

  /**
   * This defines methods for moving xml data between an XMLControl and
   * a corresponding Java object.
   *
   * @author Douglas Brown
   * @version 1.0
   */
  public interface ObjectLoader {

    /**
     * Saves data from an object to an XMLControl. The object must
     * be castable to the class control.getObjectClass().
     *
     * @param control the xml control
     * @param obj the object
     */
    public void saveObject(XMLControl control, Object obj);

    /**
     * Creates an object from data in an XMLControl. The returned object must
     * be castable to the class control.getObjectClass().
     *
     * @param control the xml control
     * @return a new object
     */
    public Object createObject(XMLControl control);

    /**
     * Loads an object with data from an XMLControl. The object must
     * be castable to the class control.getObjectClass().
     *
     * @param control the xml control
     * @param obj the object
     * @return the loaded object
     */
    public Object loadObject(XMLControl control, Object obj);
  }

  // static initializer defines loaders for commonly used classes
  static {
    setLoader(Color.class, new XML.ObjectLoader() {
      public void saveObject(XMLControl control, Object obj) {
        java.awt.Color color = (java.awt.Color)obj;
        control.setValue("red", color.getRed());
        control.setValue("green", color.getGreen());
        control.setValue("blue", color.getBlue());
        control.setValue("alpha", color.getAlpha());
      }
      public Object createObject(XMLControl control){
        int r = control.getInt("red");
        int g = control.getInt("green");
        int b = control.getInt("blue");
        int a = control.getInt("alpha");
        return new java.awt.Color(r, g, b, a);
      }
      public Object loadObject(XMLControl control, Object obj){
        int r = control.getInt("red");
        int g = control.getInt("green");
        int b = control.getInt("blue");
        int a = control.getInt("alpha");
        return new java.awt.Color(r, g, b, a);
      }
    });
    setLoader(Double.class, new XML.ObjectLoader() {
      public void saveObject(XMLControl control, Object obj) {
        Double dbl = (Double)obj;
        control.setValue("value", dbl.doubleValue());
      }
      public Object createObject(XMLControl control){
        double val = control.getDouble("value");
        return new Double(val);
      }
      public Object loadObject(XMLControl control, Object obj){
        Double dbl = (Double)obj;
        double val = control.getDouble("value");
        if (dbl.doubleValue() == val) return dbl;
        return new Double(val);
      }
    });
    setLoader(Integer.class, new XML.ObjectLoader() {
      public void saveObject(XMLControl control, Object obj) {
        Integer i = (Integer)obj;
        control.setValue("value", i.intValue());
      }
      public Object createObject(XMLControl control){
        int val = control.getInt("value");
        return new Integer(val);
      }
      public Object loadObject(XMLControl control, Object obj){
        Integer i = (Integer)obj;
        int val = control.getInt("value");
        if (i.intValue() == val) return i;
        return new Integer(val);
      }
    });
    setLoader(Boolean.class, new XML.ObjectLoader() {
      public void saveObject(XMLControl control, Object obj) {
        Boolean bool = (Boolean)obj;
        control.setValue("value", bool.booleanValue());
      }
      public Object createObject(XMLControl control){
        boolean val = control.getBoolean("value");
        return new Boolean(val);
      }
      public Object loadObject(XMLControl control, Object obj){
        Boolean bool = (Boolean)obj;
        boolean val = control.getBoolean("value");
        if (bool.booleanValue() == val) return bool;
        return new Boolean(val);
      }
    });
  }
}
