/*
 *  The org.opensourcephysics.controls package defines the framework for building
 *  user interface controls for the book Simulations in Physics.
 *  Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 *
 */
package org.opensourcephysics.controls;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.NumberFormat;
import java.util.*;

import javax.swing.*;

/**
 *  A Control class for arbitray objects.
 *
 *  The user inteface contains an input text area and a message area.
 *  Custom bottons can be added.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class TextControl extends ControlFrame {
  ParsableTextArea    inputTextArea;
  JTextArea           messageTextArea;
  JPanel              buttonPanel;
  JLabel              clearLabel;
  NumberFormat numberFormat = NumberFormat.getInstance();

  /**
   *  Constructs an TextControl.
   *
   * @param  _model
   */
  public TextControl(Object _model) {
    super("OSP Control");
    model = _model;
    Font   labelFont  = new Font("Dialog", Font.BOLD, 12);
    JLabel inputLabel = new JLabel("Enter your input below", SwingConstants.CENTER);
    inputLabel.setFont(labelFont);
    JLabel messageLabel = new JLabel("Messages", SwingConstants.CENTER);
    messageLabel.setFont(labelFont);
    inputTextArea   = new ParsableTextArea();
    messageTextArea = new JTextArea(5, 5);
    JScrollPane inputScrollPane   = new JScrollPane(inputTextArea);
    JScrollPane messageScrollPane = new JScrollPane(messageTextArea);
    Container cp         = getContentPane();
    JPanel    northPanel = new JPanel(new BorderLayout());
    northPanel.add(inputLabel, BorderLayout.NORTH);
    northPanel.add(inputScrollPane, BorderLayout.CENTER);
    buttonPanel = new JPanel();
    JPanel bottomPanel = new JPanel(new BorderLayout());
    bottomPanel.add(buttonPanel, BorderLayout.NORTH);
    bottomPanel.add(messageLabel, BorderLayout.SOUTH);
    northPanel.add(bottomPanel, BorderLayout.SOUTH);
    cp.add(northPanel, BorderLayout.NORTH);
    cp.add(messageScrollPane, BorderLayout.CENTER);
    JPanel southPanel = new JPanel(new BorderLayout());
    southPanel.addMouseListener(new ClearMouseAdapter());
    clearLabel = new JLabel(" clear");
    clearLabel.setFont(new Font(clearLabel.getFont().getFamily(), Font.PLAIN, 9));
    clearLabel.setForeground(Color.black);
    southPanel.add(clearLabel, BorderLayout.WEST);
    cp.add(southPanel, BorderLayout.SOUTH);
    messageTextArea.setEditable(false);
    inputScrollPane.setPreferredSize(new Dimension(250,200));
    messageScrollPane.setPreferredSize(new Dimension(250,50));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation( (d.width - getSize().width) / 2,(d.height - getSize().height) / 2); // center the frame
    init();
  }

  /**
   * Initializes this control after all objects have been created.
   *
   * Override this method and change the default close operation if this control is used with an applet.
   */
  protected void init(){
    validate();
    pack();
    setVisible(true);
    setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
  }

/**
 * Locks the control's interface. Values sent to the control will not
 * update the display until the control is unlocked.
 *
 * @param lock boolean
 */
  public void setLockValues(boolean lock){
    inputTextArea.setLockValues(lock);
  }


  /**
   *  Reads the current property names.
   *
   * @return      the property names
   */
  public Collection getPropertyNames() {
    return inputTextArea.getCurrentMap().keySet();
  }


  /**
   *  Creates a string representation of the control parameters.
   *
   * @return    the control parameters
   */
  public String toString() {
    return inputTextArea.getText();
  }

  /**
   *  Adds an initial value of a parameter to the input display.
   *
   * @param  par  the parameter name
   * @param  val  the initial parameter value
   */
  public void setValue(String par, Object val) {
    inputTextArea.setValue(par, val.toString());
  }

  /**
   *  Adds an initial boolean value of a parameter to the input display.
   *
   * @param  par  the parameter name
   * @param  val  the initial parameter value
   */
  public void setValue(String par, boolean val) {
    inputTextArea.setValue(par, String.valueOf(val));
  }

  /**
   *  Adds an initial value of a parameter to the input display.
   *
   * @param  par  the parameter name
   * @param  val  the initial parameter value
   */
  public void setValue(String par, double val) {
    inputTextArea.setValue(par, Double.toString(val));
  }

  /**
   *  Adds an initial value of a parameter to the input display.
   *
   * @param  par  the parameter name
   * @param  val  the initial parameter value
   */
  public void setValue(String par, int val) {
    inputTextArea.setValue(par, Integer.toString(val));
  }

  /**
   *  Reads a parameter value from the input display.
   *
   * @param  par
   * @return      double the value of of the parameter
   */
  public double getDouble(String par) {
    double val = 0;
    try {
      String str = inputTextArea.getValue(par);
      //val = Double.parseDouble(str);
      val = numberFormat.parse(str).doubleValue();
    } catch(VariableNotFoundException e) {
      println(e.getMessage());
    } catch(java.text.ParseException e) {
      println("Variable " + par + " is not a number");
    }
    return val;
  }

  /**
   *  Reads a parameter value from the input display.
   *
   * @param  par
   * @return      int the value of of the parameter
   */
  public int getInt(String par) {
    int val = (int) getDouble(par);
    return val;
  }

  /**
   * Gets the object with the specified property name.
   * Throws an UnsupportedOperationException if the named object has not been stored.
   *
   * @param name the name
   * @return the object
   */
  public Object getObject(String name)  throws UnsupportedOperationException{
     throw new UnsupportedOperationException();
  }


  /**
   *  Reads a parameter value from the input display.
   *
   * @param  par  the parameter name
   * @return      String the value of of the parameter
   */
  public String getString(String par) {
    try {
      String str = inputTextArea.getValue(par);
      return str;
    } catch(VariableNotFoundException e) {
      println(e.getMessage());
      return "";
    }
  }

  /**
   *  Reads a parameter value from the input display.
   *
   * @param  par  the parameter name
   * @return      the value of of the parameter
   */
  public boolean getBoolean(String par) {
    boolean val = false;
    try {
      String str = inputTextArea.getValue(par);
      str = str.toLowerCase().trim();
      if(str.equals("true")) {
        val = true;
      } else if(!str.equals("false")) {
        println("Error: Boolean variable must be true or false.");
      }
    } catch(VariableNotFoundException e) {
      println(e.getMessage());
    }
    return val;
  }

  /**
   *  Adds a custom button to the control's frame.
   *
   * @param  methodName  the name of the method; the method has no parameters
   * @param  text        the button's text label
   * @return             the custom button
   */
  public JButton addButton(String methodName, String text) {
    return addButton(methodName, text, null);
  }

  /**
   *  Adds a custom button to the control's frame.
   *
   * @param  methodName   the name of the method; the method has no parameters
   * @param  text         the button's text label
   * @param  toolTipText  the button's tool tip text
   * @return              the custom button
   */
  public JButton addButton(String methodName, String text, String toolTipText) {
    JButton b = new JButton(text);
    b.setToolTipText(toolTipText);
    Class[] parameters = {};
    try {
      final java.lang.reflect.Method m = model.getClass().getMethod(methodName, parameters);
      b.addActionListener(new ActionListener() {

        public void actionPerformed(ActionEvent e) {
          Object[] args = {};
          try {
            m.invoke(model, args);
          } catch(IllegalAccessException iae) {}
          catch(java.lang.reflect.InvocationTargetException ite) {}
        }
      });
      buttonPanel.add(b);
      validate();
      pack();
    } catch(NoSuchMethodException nsme) {
      System.err.println("Error adding custom button " + text + ". The method " + methodName + "() does not exist.");
    }
    return b;
  }


  /**
   *  Prints a line of text in the message area.
   *
   * @param  s
   */
  public void println(String s) {
    print(s + "\n");
  }

  /**
   *  Prints a blank line in the message area.
   */
  public void println() {
    print("\n");
  }

  /**
   *  Prints text in the message area.
   *
   * @param  s
   */
  public void print(final String s) {
    if (SwingUtilities.isEventDispatchThread()) {
      messageTextArea.append(s);
      return;
    }
    Runnable doLater = new Runnable() {
      public void run() {
        messageTextArea.append(s);
      }
    };
    // Update from the event queue.
    java.awt.EventQueue.invokeLater(doLater);

  }

  /**
   *  Remove all text from the message area.
   */
  public void clearMessages() {
    if(SwingUtilities.isEventDispatchThread()){
      messageTextArea.setText("");
      return;
    }
    Runnable doLater = new Runnable() {
      public void run() {
        messageTextArea.setText("");
      }
    };
    // Update from the event queue.
    java.awt.EventQueue.invokeLater(doLater);
  }

  /**
   *  Remove all text from the data input area.
   */
  public void clearValues() {
    if (SwingUtilities.isEventDispatchThread()) {
      inputTextArea.setText("");
      return;
    }
    Runnable doLater = new Runnable() {
      public void run() {
        inputTextArea.setText("");
      }
    };
    // Update from the event queue.
    java.awt.EventQueue.invokeLater(doLater);
  }

  /**
   *  A signal that a method has completed. A message is printed in the message area.
   *
   * @param  message
   */
  public void calculationDone(String message) {
    // not implemented
    println(message);
  }

  class ClearMouseAdapter extends java.awt.event.MouseAdapter {

    /**
     * Method mousePressed
     *
     * @param evt
     */
    public void mousePressed(java.awt.event.MouseEvent evt) {
      clearMessages();
    }

    /**
     * Method mouseEntered
     *
     * @param evt
     */
    public void mouseEntered(java.awt.event.MouseEvent evt) {
      clearLabel.setFont(new Font(clearLabel.getFont().getFamily(), Font.BOLD, 10));
      clearLabel.setText(" click here to clear messages");
    }

    /**
     * Method mouseExited
     *
     * @param evt
     */
    public void mouseExited(java.awt.event.MouseEvent evt) {
      clearLabel.setFont(new Font(clearLabel.getFont().getFamily(), Font.PLAIN, 9));
      clearLabel.setText(" clear");
    }
  }

  /**
   * Returns an XML.ObjectLoader to save and load data for this object.
   *
   * @return the object loader
   */
  public static XML.ObjectLoader getLoader() {
    return new TextControlLoader();
  }

  /**
 * A class to save and load data for TextControls.
 */
static class TextControlLoader implements XML.ObjectLoader {
  /**
   * Saves object data to an XMLControl.
   *
   * @param control the control to save to
   * @param obj the object to save
   */
  public void saveObject(XMLControl control, Object obj) {
    TextControl cf = (TextControl)obj;
    // save parameters
    Map map = cf.inputTextArea.getCurrentMap();
    Iterator it = map.keySet().iterator();
    while (it.hasNext()) {
      String variable = (String) it.next();
      control.setValue(variable, map.get(variable));
    }
    // save the model if the control is the top level
    if (control.getLevel() == 0) {
      control.setValue("model", cf.model);
    }
  }

  /**
   * Creates an object using data from an XMLControl.
   *
   * @param control the control
   * @return the newly created object
   */
  public Object createObject(XMLControl control) {
    return new TextControl(null);
  }

  /**
   * Loads an object with data from an XMLControl.
   *
   * @param control the control
   * @param obj the object
   * @return the loaded object
   */
  public Object loadObject(XMLControl control, Object obj) {
    TextControl cf = (TextControl) obj;
    // iterate over properties and add strings to input text area
    Iterator it = control.getPropertyNames().iterator();
    cf.inputTextArea.setLockValues(true);
    while (it.hasNext()) {
      String variable = (String) it.next();
      if (control.getPropertyType(variable).equals("string")) {
        cf.inputTextArea.setValue(variable, control.getString(variable));
      }
    }
    cf.inputTextArea.setLockValues(false);
    XMLControl child = control.getChildControl("model");
    if (child != null) cf.model = child.loadObject(cf.model);
    return obj;
  }
}

}
