package org.opensourcephysics.controls;

import java.awt.event.ActionEvent;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * An AnimationControl that controls the editing of parameters.
 *
 * @author Wolfgang Christian
 * @version 1.0
 */

public class SimulationControl extends AnimationControl{

  Set fixedParameters = new HashSet();

  /**
   * Constructs a SIPAnimationControl for the given animation.
   * @param animation Animation
   */
  public SimulationControl(Simulation animation) {
    super(animation);
  }


  /**Sets the fixed property of the given parameter.
   * Fixed parameters and can only be changed during initialization.
   */

  public void setParameterToFixed(String name, boolean fixed){
    if(fixed) this.fixedParameters.add(name);
    else this.fixedParameters.remove(name);
    table.repaint();
  }

  /**
   * Determines if the given parameter is fixed and can only be changed during initialization.
   * @param name String
   * @return boolean
   */
  public boolean isParamterFixed(String name){
    return fixedParameters.contains(name);
  }

  /**
   * Stores an object in the control
   * that can only be edited when the control is in initialization mode.
   *
   * @param name
   * @param val
   */
  public void setValue(String name, Object val) {
    super.setValue( name, val);
    fixedParameters.add(name);
  }

  /**
 * Stores an object in the control that can be edited after initialization.
 *
 * @param name
 * @param val
 */
  public void setAdjustableValue(String name, Object val){
    super.setValue(name, val);
  }

  /**
   * Stores a name and a double value in the control
   * that can only be edited when the control is in initialization mode.
   *
   * @param name
   * @param val
   */
  public void setValue(String name, double val) {
    super.setValue(name, val);
    fixedParameters.add(name);
  }

  /**
   * Stores a double in the control that can be edited after initialization.
   *
   * @param name
   * @param val
   */
  public void setAdjustableValue(String name, double val) {
    super.setValue(name, val);
  }

  /**
   * Stores a name and an integer value in the control
   * that can only be edited when the control is in initialization mode.
   *
   * @param name
   * @param val
   */
  public void setValue(String name, int val) {
    super.setValue(name, val);
    fixedParameters.add(name);
  }

  /**
   * Stores an integer in the control that can be edited after initialization.
   *
   * @param name
   * @param val
   */
  public void setAdjustableValue(String name, int val) {
    super.setValue(name, val);
  }

  /**
   * Stores a name and a boolean value in the control
   * that can only be edited when the control is in initialization mode.
   *
   * @param name
   * @param val
   */
  public void setValue(String name, boolean val) {
    super.setValue(name, val);
    fixedParameters.add(name);
  }

  /**
 * Removes a parameter from this control.
 *
 * @param name
 */
public void removeParameter(String name) {
  super.removeParameter(name);
  fixedParameters.remove(name);
}


  /**
   * Stores a boolean in the control that can be edited after initialization.
   *
   * @param name
   * @param val
   */
  public void setAdjustableValue(String name, boolean val) {
    super.setValue(name, val);
  }


  /**
   * Resets the control by putting the control into its initialization state.
   *
   * Fixed parameters are editiable when the control is in the initialization state.
   *
   * @param e
   */
  void resetBtnActionPerformed(ActionEvent e) {
    Iterator it = fixedParameters.iterator();
    while (it.hasNext()) {
      String par = (String) it.next();
      table.tableModel.setParameterEditable(par, true);
    }
    table.repaint();
    super.resetBtnActionPerformed(e);
  }

  /**
   * Starts, stops, or steps the animation depending on the mode.
   *
   * @param e
   */
  void startBtnActionPerformed(ActionEvent e) {
    boolean doEndStepping=false;
    table.getDefaultEditor(Object.class).stopCellEditing();
    if (e.getActionCommand().equals(ControlsRes.ANIMATION_INIT)) {
      // animation is being initialized and fixed parameters are no longer editable
      Iterator it = fixedParameters.iterator();
      while (it.hasNext()) {
        String par = (String) it.next();
        table.tableModel.setParameterEditable(par, false);
      }
      table.repaint();
    }
    else if (e.getActionCommand().equals(ControlsRes.ANIMATION_START)) {
      // notify the model that the animation thread is about start
       ((Simulation)model).startRunning();
    }
    else { // action command = Stop
       doEndStepping=true; //delay until after the tread has died
    }
    super.startBtnActionPerformed(e);
    if(doEndStepping)((Simulation)model).stopRunning();
  }


  /**
   * Steps the animation and notifies the model that a step has begun and ended.
   *
   * @param e
   */
  void stepBtnActionPerformed(ActionEvent e) {
    ((Simulation)model).startRunning();
     super.stepBtnActionPerformed(e);
    ((Simulation)model).stopRunning();
  }

  /**
   * Creates a SIP animation control and establishes communication between the control and the model.
   *
   * @param model SIPAnimation
   * @return AnimationControl
   */
  public static SimulationControl createApp(Simulation model){
    SimulationControl control = new SimulationControl(model);
    model.setControl (control);
    return control;
  }

  /**
 * Creates a simulation control and establishes communication between the control and the model.
 * Initial parameters are set using the xml data.
 *
 * @param model Simulation
 * @param xml String[]
 * @return SimulationControl
 */
public static SimulationControl createApp(Simulation model, String[] xml) {
  SimulationControl control = createApp(model);
  control.loadXML(xml);
  return control;
}


}
