/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.awt.Font;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.swing.JTextArea;

/**
 * Title:        ParsableTextArea
 * Description:  JTextArea with the ability to parse parameters.
 */
public class ParsableTextArea extends JTextArea {
  HashMap pendingMap = new LinkedHashMap();
  HashMap currentMap = new LinkedHashMap();
  HashMap lockedMap = new LinkedHashMap();
  boolean locked = false;

  /**
   * ParsableTextArea constructor.
   *
   */
  public ParsableTextArea() {
    super(10, 10);
    setFont(new Font("Monospaced", Font.PLAIN, 12));
  }

  /**
   * Gets the stored string parameter associated with the variable name.
   *
   * @param variable
   *
   * @return the string
   *
   * @throws VariableNotFoundException
   */
  public String getValue(String variable) throws VariableNotFoundException {
    if (locked)synchronized (lockedMap) {
      String val = (String) lockedMap.get(variable);
      if (val != null)return (String) lockedMap.get(variable);
      throw new VariableNotFoundException("Variable " + variable +
                                          " not found.");
    }
    synchronized (currentMap) {
      updateCurrentMap(); // gets existing values
      synchronized (pendingMap) {
        currentMap.putAll(pendingMap); // add pending values
      }
      String val = (String) currentMap.get(variable);
      if (val != null)return val;
    }
    throw new VariableNotFoundException("Variable " + variable + " not found.");
  }

  /**
   * Locks values to those currently on display.
   * @param lock boolean
   */
  public synchronized void setLockValues(boolean lock) {
    if (locked == lock)return; // no change so don't do anything
    locked = lock;
    if (locked) { // get all current and pending values
      synchronized (lockedMap) {
        lockedMap.clear();
        synchronized (currentMap) {
          updateCurrentMap();
          lockedMap.putAll(currentMap); // add current values
        }
        synchronized (pendingMap) {
          lockedMap.putAll(pendingMap); // add pending values
        }
      }
    }
    else { // control has just been unlocked
      setValue(null, null); // updates the text area
    }
  }

  /**
   * Stores a variable name and an associated value.
   *
   * @param variable
   * @param val
   */
  public void setValue(String variable, String val) {
    Runnable doLater = new Runnable() {
      public void run() {
        updateText();
      }
    };
    if (variable != null)synchronized (pendingMap) {
      pendingMap.put(variable, val);
    }
    if (locked && variable != null)synchronized (lockedMap) {
      lockedMap.put(variable, val);
    }
    if (!locked) java.awt.EventQueue.invokeLater(doLater);
  }

  public HashMap getCurrentMap(){
    synchronized (currentMap) {
      updateCurrentMap(); // gets existing values
      synchronized (pendingMap) {
        currentMap.putAll(pendingMap); // add pending values
      }
      return (HashMap)currentMap.clone();
    }
  }

  private synchronized void updateText() { // should only be called from event queue
    synchronized (currentMap) {
      synchronized (pendingMap) {
        if (pendingMap.size() == 0)return;
        updateCurrentMap(); // puts existing variables into currentMap
        currentMap.putAll(pendingMap);
        pendingMap.clear();
      }
      Set set = currentMap.keySet();
      Iterator it = set.iterator();
      StringBuffer newText = new StringBuffer(set.size() * 25);
      while (it.hasNext()) {
        String variable = (String) it.next();
        newText.append(variable);
        newText.append('=');
        newText.append(currentMap.get(variable));
        newText.append('\n');
      }
      setText(newText.toString()); // access text area because we are in event queue
    }
  }

  private void updateCurrentMap() {
    currentMap.clear();
    String text = getText();
    StringTokenizer st = new StringTokenizer(text, "\n");
    while (st.hasMoreTokens()) {
      String aLine = st.nextToken().trim();
      int index = aLine.indexOf("=");
      if ( (aLine != null) && (index != -1)) {
        currentMap.put(aLine.subSequence(0, index),
                       aLine.subSequence(index + 1, aLine.length()));
      }
    }
  }

  /**
   * Returns an XML.ObjectLoader to save and load data for this object.
   *
   * @return the object loader
   */
  public static XML.ObjectLoader getLoader() {
    return new ParsableTextAreaLoader();
  }

  /**
   * A class to save and load data for OSPControls.
   */
  static class ParsableTextAreaLoader implements XML.ObjectLoader {
    /**
     * Saves object data to an ObjectElement.
     *
     * @param element the element to save to
     * @param obj the object to save
     */
    public void saveObject(XMLControl element, Object obj) {
      Map map = ((ParsableTextArea) obj).getCurrentMap();
      Iterator it = map.keySet().iterator();
      while (it.hasNext()) {
        String variable = (String) it.next();
        element.setValue(variable, map.get(variable));
      }
    }

    /**
     * Creates an object using data from an ObjectElement.
     *
     * @param element the element
     * @return the newly created object
     */
    public Object createObject(XMLControl element) {
      return new ParsableTextArea();
    }

    /**
     * Loads an object with data from an ObjectElement.
     *
     * @param element the element
     * @param obj the object
     * @return the loaded object
     */
    public Object loadObject(XMLControl element, Object obj) {
      ParsableTextArea pta = (ParsableTextArea) obj;
      // iterate over properties and add them to pts
      Iterator it=element.getPropertyNames().iterator();
      pta.setLockValues(true);
      while (it.hasNext()) {
        String variable = (String) it.next();
        pta.setValue(variable,element.getString(variable));
      }
      pta.setLockValues(false);
      return obj;
    }
  }

}
