/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

/**
 * This modal dialog lets the user choose any number of items
 * from a supplied list. Items not selected are removed from the list.
 */
public class ListChooser extends JDialog {

  // instance fields
  private JPanel checkPane = new JPanel();
  private Object[] objects;
  private boolean[] selections;
  private JCheckBox[] checkBoxes;
  private boolean applyChanges = false;

  /**
   * Constructs a dialog with the specified title and text.
   *
   * @param title the title of the dialog
   * @param text the label text
   */
  public ListChooser(String title, String text) {
    this(title, text, null);
  }

  /**
   * Constructs a dialog with the specified title, text and owner.
   *
   * @param title the title of the dialog
   * @param text the label text
   * @param owner the component that owns the dialog (may be null)
   */
  public ListChooser(String title, String text, Component owner) {
    super(JOptionPane.getFrameForComponent(owner), true);
    setTitle(title);
    JLabel textLabel = new JLabel(" " + text);
    // create the buttons
    JButton cancelButton = new JButton("Cancel");
    JButton okButton = new JButton("OK");
    JButton selectAllButton = new JButton("Select_All");
    cancelButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        setVisible(false);
      }
    });
    okButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        for (int i = 0; i < checkBoxes.length; i++) {
          selections[i] = checkBoxes[i].isSelected();
        }
        applyChanges = true;
        setVisible(false);
      }
    });
    selectAllButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        for (int i = 0; i < checkBoxes.length; i++) {
          checkBoxes[i].setSelected(true);
        }
      }
    });
    getRootPane().setDefaultButton(okButton);
    // lay out the header pane
    JPanel headerPane = new JPanel();
    headerPane.setLayout(new BoxLayout(headerPane, BoxLayout.X_AXIS));
    headerPane.add(textLabel);
    headerPane.add(Box.createHorizontalGlue());
    headerPane.add(selectAllButton);
    headerPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
    // lay out the scroll pane
    checkPane.setLayout(new BoxLayout(checkPane, BoxLayout.Y_AXIS));
    checkPane.setBackground(Color.white);
    JScrollPane scroller = new JScrollPane(checkPane);
    scroller.setPreferredSize(new Dimension(250, 180));
    JPanel scrollPane = new JPanel(new BorderLayout());
    scrollPane.add(scroller, BorderLayout.CENTER);
    scrollPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
    // lay out the button pane
    JPanel buttonPane = new JPanel();
    buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
    buttonPane.setBorder(BorderFactory.createEmptyBorder(0, 10, 10, 10));
    buttonPane.add(Box.createHorizontalGlue());
    buttonPane.add(cancelButton);
    buttonPane.add(Box.createRigidArea(new Dimension(10, 0)));
    buttonPane.add(okButton);
    // add everything to the content pane
    Container contentPane = getContentPane();
    contentPane.add(headerPane, BorderLayout.NORTH);
    contentPane.add(scrollPane, BorderLayout.CENTER);
    contentPane.add(buttonPane, BorderLayout.SOUTH);
    pack();
    // center dialog on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - this.getBounds().width) / 2;
    int y = (dim.height - this.getBounds().height) / 2;
    setLocation(x, y);
  }

  /**
   * Allows the user to choose from the supplied list. Items not selected are
   * removed from the list.
   *
   * @param choices a list of objects to choose from
   * @param names an optional list of descriptive names
   * @return <code>true</code> if OK button was clicked
   */
  public boolean choose(java.util.Collection choices, java.util.Collection names) {
    checkPane.removeAll();
    checkBoxes = new JCheckBox[choices.size()];
    selections = new boolean[choices.size()];
    objects = new Object[choices.size()];
    ArrayList nameList = new ArrayList();
    if (names != null) {
      nameList.addAll(names);
    }
    Iterator it = choices.iterator();
    int i = 0;
    while (it.hasNext()) {
      objects[i] = it.next();
      selections[i] = false;
      if (nameList.get(i) == null) {
        checkBoxes[i] = new JCheckBox(objects[i].toString());
      }
      else {
        checkBoxes[i] = new JCheckBox(nameList.get(i).toString());
      }
      checkBoxes[i].setBackground(Color.white);
      checkPane.add(checkBoxes[i]);
      i++;
    }
    setVisible(true);
    if (!applyChanges) return false;
    for (i = 0; i < objects.length; i++) {
      if (!selections[i]) {
        choices.remove(objects[i]);
      }
    }
    return true;
  }

}
