/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.text.NumberFormat;
import java.util.HashMap;

/**
 * A Control without a graphical user interface.
 *
 * @author       Joshua Gould
 * @version 1.0
 */
public class HiddenControl implements Control {

  HashMap map = new HashMap();
  NumberFormat numberFormat = NumberFormat.getInstance();

  /**
   * Locks the control's interface. Values sent to the control will not
   * update the display until the control is unlocked.
   *
   * @param lock boolean
   */
  public void setLockValues(boolean lock) {
    // this control does not have a user interface
  }

   /**
	 *  Adds an initial value of a parameter to the input display. Input parameters
	 *  should be read when the calculation is performed.
	 *
	 *@param  par  the parameter name
	 *@param  val  the initial parameter value
	 *@since
	 */
	public void setValue(String par, Object val) {
		map.put(par, val.toString());
	}


	/**
	 *  Add an initial boolean value of a parameter to the input display. Input
	 *  parameters should be read when the calculation is performed.
	 *
	 *@param  par  the parameter name
	 *@param  val  the initial parameter value
	 *@since
	 */
	public void setValue(String par, boolean val) {
		map.put(par, String.valueOf(val));
	}


	/**
	 *  Add an initial value of a parameter to the input display. Input parameters
	 *  should be read when the calculation is performed.
	 *
	 *@param  par  the parameter name
	 *@param  val  the initial parameter value
	 *@since
	 */
	public void setValue(String par, double val) {
		map.put(par, Double.toString(val));
	}


	/**
	 *  Add an initial value of a parameter to the input display. Input parameters
	 *  should be read when the calculation is performed.
	 *
	 *@param  par  the parameter name
	 *@param  val  the initial parameter value
	 *@since
	 */
	public void setValue(String par, int val) {
		map.put(par, Integer.toString(val));
	}

	public void scriptValue(String par, String val) {
		map.put(par, val);
	}

        /**



  /**
   * Gets the names of all properties stored in this control.
   *
   * @return List
   */
  public java.util.Collection getPropertyNames() {
     return map.keySet();
  }



	/**
	 *  Read a parameter value from the input display.
	 *
	 *@param  par
	 *@return      double the value of of the parameter
	 *@since
	 */
        public double getDouble(String par) {
          String str = getString(par);
          if (str.equals("")) {
            return 0;
          }
          try {
            double val = numberFormat.parse(str).doubleValue();
            return val;
          } catch (java.text.ParseException e) {
            println("Variable " + par + " is not a number");
            return 0;
          }
        }


	/**
	 *  Read a parameter value from the input display.
	 *
	 *@param  par
	 *@return      int the value of of the parameter
	 *@since
	 */
	public int getInt(String par) {
		int val = (int) getDouble(par);
		return val;
	}



        /**
         * Gets the object with the specified property name.
         * Throws an UnsupportedOperationException if the named object has not been stored.
         *
         * @param name the name
         * @return the object
         */
        public Object getObject(String name){
            return map.get(name);
        }


	/**
	 *  Reads a parameter value from the input display.
	 *
	 *@param  par  the parameter name
	 *@return      String the value of of the parameter
	 *@since
	 */
	public String getString(String par) {
		String str = (String) map.get(par);
		if(str == null) {
			println("Variable " + par + " not found.");
			return "";
		}
		return str;
	}


	/**
	 *  Read a parameter value from the input display.
	 *
	 *@param  par  the parameter name
	 *@return      the value of of the parameter
	 *@since
	 */
	public boolean getBoolean(String par) {
		String str = getString(par);
		if(str.equals("")) {
			return false;
		}
		str = str.toLowerCase().trim();
		if(str.equals("true")) {
			return true;
		}
		if(str.equals("false")) {
			return false;
		}
		println("Error: Boolean variable must be true or false.");
		return false;
	}


   public void println (String s) {
      System.out.println(s);
   }


   public void println () {
      System.out.println();
   }


   public void print (String s) {
      System.out.println(s);
   }


   public void clearMessages () {}


   public void calculationDone (String s) {}

	public void clearValues(){}
}



