/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.table.AbstractTableModel;
import org.opensourcephysics.numerics.DoubleArray;
import org.opensourcephysics.numerics.IntegerArray;

/**
 * A Control that shows name-value pairs in a JTable
 *
 * @author  W. Christian
 * @version 1.0
 */
public class ControlTableModel extends AbstractTableModel implements Control {

   static final Color ERROR_RED = new Color(255, 128, 128);
   static final Color PANEL_BACKGROUND = javax.swing.UIManager.getColor("Panel.background");
   static final int NO_ERROR = 0;
   static final int DOUBLE_ERROR = 1;
   static final int INTEGER_ERROR = 2;
   static final int INDEX_ERROR = 3;
   ArrayList names = new ArrayList(16);
   ArrayList values = new ArrayList(16);
   ArrayList colors = new ArrayList(16);
   ArrayList errors = new ArrayList(16);
   ArrayList editList = new ArrayList(16);
   ArrayList inspectorList = new ArrayList(16);
   boolean locked = false;
   boolean enabled = true; // user can edit table values
   int setValueError = NO_ERROR;

   /**
    * Locks the control's interface. Values sent to the control will not
    * update the display until the control is unlocked.
    *
    * @param lock boolean
    */
   public void setLockValues(boolean lock) {
      locked = lock;
      if(locked) {
         // don't change the table
      } else {
         fireTableDataChanged();
      }
   }

   /**
    * Gets the names of all properties stored in this control.
    *
    * @return List
    */
   public java.util.Collection getPropertyNames() {
      return(java.util.List) names.clone();
   }

   /**
    *  Adds an initial value of a parameter to the input display. Input parameters
    *  should be read when the calculation is performed.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    * @since
    */
   public void setValue(String name, Object val) {
      if(val instanceof String) {
         addToArrays(name, new Expression(""+val));
      } else if(val instanceof double[]) {
         addToArrays(name, new DoubleArray(((double[]) val)));
      } else if(val instanceof int[]) {
         addToArrays(name, new IntegerArray(((int[]) val)));
      } else {
         addToArrays(name, val);
      }
   }

   /**
    *  Adds an initial boolean value of a parameter to the input display. Input
    *  parameters should be read when the calculation is performed.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    * @since
    */
   public void setValue(String name, boolean val) {
      addToArrays(name, Boolean.valueOf(val));
   }

   /**
    *  Adds an initial value of a parameter to the input display. Input parameters
    *  should be read when the calculation is performed.
    *
    * @param  name  the parameter name
    * @param  val  the initial parameter value
    * @since
    */
   public void setValue(String name, double val) {
      addToArrays(name, new Expression(""+val));
   }

   /**
    *  Adds a parameter to the table model.
    *
    * @param  name  the parameter name
    * @param  val  the initial parameter value
    * @since
    */
   public void setValue(String name, int val) {
      addToArrays(name, new Expression(""+val));
   }

   /**
    *  Removes a parameter from the table.
    *
    * @param  name  the parameter name
    */
   public void removeParameter(String name) {
      removeFromArrays(name);
   }

   private synchronized void addToArrays(Object name, Object val) {
      if((name==null)||(val==null)) {
         return;
      }
      OSPInspector inspector = OSPInspector.getInspector(val);
      if(names.contains(name)) {                     // change existing parameter
         int index = names.indexOf(name);
         //names.set(index, name);
         values.set(index, val);
         errors.set(index, new Boolean(false));
         inspectorList.set(index, inspector);
         if(inspector!=null) {
            colors.set(index, inspector.highlightColor);
            editList.set(index, new Boolean(false)); //use the inspector to edit the object
         } else {
            colors.set(index, Color.WHITE);
            //editList.set(index, new Boolean(true));  // table entries can be edited by default
         }
         if(!locked) {
            fireTableDataChanged();
         }
      } else {                                       // add new parameter
         names.add(name);
         values.add(val);
         errors.add(new Boolean(false));
         editList.add(new Boolean(true));            //use the inspector to edit the object
         inspectorList.add(inspector);
         if(inspector!=null) {
            colors.add(inspector.highlightColor);
            editList.add(new Boolean(true));         // table entries can be edited by default
         } else {
            colors.add(Color.WHITE);
            editList.add(new Boolean(true));         // table entries can be edited by default
         }
         fireTableStructureChanged();
      }
   }

   private synchronized void removeFromArrays(Object name) {
      if(names.contains(name)) { // change existing parameter
         int index = names.indexOf(name);
         names.remove(index);
         values.remove(index);
         colors.remove(index);
         errors.remove(index);
         if(!locked) {
            fireTableStructureChanged();
         }
      }
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par
    * @return      double the value of of the parameter
    * @since
    */
   public double getDouble(String par) {
      if(!names.contains(par)) {
         println("Variable "+par+" not found.");
         OSPLog.info("Variable "+par+" not found in table control.");
         return Double.NaN;
      }
      int index = names.indexOf(par);
      Object obj = values.get(index);
      if(obj.getClass()==Expression.class) {
         return((Expression) obj).toDouble(index);
      } else if(obj instanceof Double) {
         return((Double) obj).doubleValue();
      }
      return Double.NaN;
   }

   /**
    *  Reads an integer value from the table.
    *
    * @param  par
    * @return      int the value of of the parameter
    * @since
    */
   public int getInt(String par) {
      if(!names.contains(par)) {
         // println("Variable " + par + " not found in table control.");
         OSPLog.fine("Variable "+par+" not found in table control.");
         return 0;
      }
      int index = names.indexOf(par);
      Object obj = values.get(index);
      if(obj.getClass()==Expression.class) {
         return(int) ((Expression) obj).toDouble(index);
      } else if(obj instanceof Integer) {
         return((Integer) obj).intValue();
      }
      return 0;
   }

   /**
    * Gets the object with the specified property name.
    * Throws an UnsupportedOperationException if the named object has not been stored.
    *
    * @param name the name
    * @return the object
    */
   public synchronized Object getObject(String name) {
      int index = names.indexOf(name);
      Object obj = values.get(index);
      if(obj.getClass()==DoubleArray.class) {
         DoubleArray da = ((DoubleArray) obj);
         if(da.getError()>0) {
            colors.set(index, ERROR_RED);
            errors.set(index, new Boolean(true));
            fireTableDataChanged();
            return da.getArray();
         }
         colors.set(index, Color.WHITE);
         errors.set(index, new Boolean(false));
         return da.getArray();
      } else if(obj.getClass()==IntegerArray.class) {
         IntegerArray da = ((IntegerArray) obj);
         if(da.getError()>0) {
            colors.set(index, ERROR_RED);
            errors.set(index, new Boolean(true));
            fireTableDataChanged();
            return da.getArray();
         }
         colors.set(index, Color.WHITE);
         errors.set(index, new Boolean(false));
         return da.getArray();
      } else {
         return obj;
      }
   }

   /**
    *  Reads a string value from the input table model.
    *
    * @param  par  the parameter name
    * @return      String the value of of the parameter
    * @since
    */
   public String getString(String par) {
      if(!names.contains(par)) {
         println("Variable "+par+" not found.");
         OSPLog.info("Variable "+par+" not found in table control.");
         return "";
      }
      int index = names.indexOf(par);
      return values.get(index).toString();
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par  the parameter name
    * @return      the value of of the parameter
    * @since
    */
   public boolean getBoolean(String par) {
      if(!names.contains(par)) {
         println("Variable "+par+" not found.");
         OSPLog.info("Variable "+par+" not found in table control.");
         return false;
      }
      int index = names.indexOf(par);
      Object obj = values.get(index);
      if(obj instanceof Boolean) {
         return((Boolean) obj).booleanValue();
      } else if(obj.toString().trim().toLowerCase().equals("false")) {
         return false;
      } else if(obj.toString().trim().toLowerCase().equals("true")) {
         return true;
      } else {
         println("Variable "+par+" is not a boolean.");
         return false;
      }
   }

   public void println(String s) {
      OSPLog.fine(s);
      System.out.println(s);
   }

   public void println() {
      System.out.println();
   }

   public void print(String s) {
      OSPLog.fine(s);
      System.out.println(s);
   }

   public void clearMessages() {}

   public void calculationDone(String s) {
      OSPLog.fine("Calculation Done: "+s);
   }

   public void clearValues() {}

   /**
    * getColumnCount
    *
    * @return int
    */
   public int getColumnCount() {
      return 2;
   }

   /**
    * getRowCount
    *
    * @return int
    */
   public synchronized int getRowCount() {
      return names.size();
   }

   public boolean isCellEditable(int rowIndex, int columnIndex) {
      if((columnIndex==1)&&enabled) { // selected value can be edited
         return((Boolean) editList.get(rowIndex)).booleanValue();
      } else {
         return false;
      }
   }

   /**
    * Sets the object value for the cell at <code>column</code> and
    * <code>row</code>.  <code>aValue</code> is the new value.  This method
    * will generate a <code>tableChanged</code> notification.
    *
    * @param   aValue          the new value; this can be null
    * @param   row             the row whose value is to be changed
    * @param   column          the column whose value is to be changed
    * @exception  ArrayIndexOutOfBoundsException  if an invalid row or
    *               column was given
    */
   public void setValueAt(Object aValue, int row, int column) {
      setValueError = NO_ERROR;
      if((column>1)||(row>=names.size())||(row<0)||(column<0)) {
         setValueError = INDEX_ERROR;
         return;
      }
      if(column==0) { // first column of table model contains names
         names.set(row, aValue);
         return;
      }
      // we are in the values column
      Object val = values.get(row);
      if(val instanceof Boolean) {
         values.set(row, Boolean.valueOf(aValue.toString()));
      } else if(val.getClass()==Expression.class) {
         double lastVal = ((Expression) val).lastVal;
         Expression exp = new Expression(aValue.toString());
         exp.lastVal = lastVal;
         values.set(row, exp);
      } else if(val.getClass()==DoubleArray.class) {
         DoubleArray lastVal = ((DoubleArray) val);
         DoubleArray newVal = new DoubleArray(aValue.toString());
         if(newVal.getError()>0) {
            newVal.setDefaultArray(lastVal.getArray());
         }
         values.set(row, newVal);
      } else if(val.getClass()==IntegerArray.class) {
         IntegerArray lastVal = ((IntegerArray) val);
         IntegerArray newVal = new IntegerArray(aValue.toString());
         if(newVal.getError()>0) {
            newVal.setDefaultArray(lastVal.getArray());
         }
         values.set(row, newVal);
      } else if(val.getClass()==Double.class) {
         try {
            Double newVal = new Double(aValue.toString());
            values.set(row, newVal);
         } catch(NumberFormatException ex) {
            setValueError = DOUBLE_ERROR;
         }
      } else if(val.getClass()==Integer.class) {
         try {
            Integer newVal = new Integer(aValue.toString());
            values.set(row, newVal);
         } catch(NumberFormatException ex) {
            setValueError = INTEGER_ERROR;
         }
      } else {
         values.set(row, aValue);
      }
      return;
   }

   /**
    *  Gets the cell color;
    *
    * @param rowIndex int
    * @param columnIndex int
    * @return Color
    */
   public Color getColorAt(int row, int column) {
      if((column>1)||(row>=colors.size())||(row<0)) {
         return Color.WHITE;
      }
      if(column==0) {
         return PANEL_BACKGROUND;
      } else {
         return(Color) colors.get(row);
      }
   }

   /**
    * Gets the error flag for the given row.
    *
    * @param index int
    * @return boolean
    */
   public boolean hasError(int index) {
      return((Boolean) errors.get(index)).booleanValue();
   }

   /**
    * Sets the error flag for the given row.
    * @param index int
    * @param err boolean
    */
   public void setError(int index, boolean err) {
      errors.set(index, new Boolean(err));
   }

   /**
    *  Sets the cell color;
    *
    * @param rowIndex int
    * @param color Color
    */
   public void setColorAt(int row, Color color) {
      if((row>=colors.size())||(row<0)) {
         return;
      }
      colors.set(row, color);
   }

   /**
    * Sets the editable value for the given parameter.
    * @param par String
    * @param editable boolean
    */
   void setParameterEditable(String par, boolean editable) {
      int index = names.indexOf(par);
      editList.set(index, new Boolean(editable));
   }

   /**
    * Enables the table model so that cells are editable.
    * @param enabled boolean
    */
   void setEnabled(boolean enabled) {
      this.enabled = enabled;
   }

   /**
    * Gets a value for a cell.
    *
    * @param rowIndex int
    * @param columnIndex int
    * @return Object
    */
   public Object getValueAt(int row, int column) {
      if((column>1)||(row>=names.size())) {
         return null;
      }
      if(column==0) { // first column of table model contains names
         return names.get(row);
      }
      return values.get(row);
   }

   public String getColumnName(int c) {
      if(c==0) {
         return "name";
      } else if(c==1) {
         return "value";
      } else {
         return super.getColumnName(c);
      }
   }

   class Expression {

      String str;
      double val;
      double lastVal = 0;
      Expression(double val) {
         this.val = val;
         lastVal = val;
         str = ""+val;
      }

      Expression(String str) {
         this.str = str;
         val = Double.NaN;
         try {
            val = Double.parseDouble(str);
         } catch(NumberFormatException ex) {}
         if(Double.isNaN(val)) {
            val = org.opensourcephysics.numerics.Util.evalMath(str);
         }
         if(Double.isNaN(val)) {
            lastVal = val;
         }
      }

      public String toString() {
         return str;
      }

      public double toDouble(int index) {
         if(Double.isNaN(val)) {
            colors.set(index, ERROR_RED);
            errors.set(index, new Boolean(true));
            fireTableDataChanged();
            return lastVal;
         }
         return val;
      }
   }

   /**
    * Returns an XML.ObjectLoader to save and load data for this object.
    *
    * @return the object loader
    */
   public static XML.ObjectLoader getLoader() {
      return new ControlTableModelLoader();
   }

   /**
    * A class to save and load data for OSPControls.
    */
   static class ControlTableModelLoader implements XML.ObjectLoader {

      /**
       * Saves object data to an ObjectElement.
       *
       * @param element the element to save to
       * @param obj the object to save
       */
      public void saveObject(XMLControl element, Object obj) {
         ArrayList names = (ArrayList) ((ControlTableModel) obj).names.clone();
         ArrayList values = (ArrayList) ((ControlTableModel) obj).values.clone();
         for(int i = 0, n = names.size(); i<n; i++) {
            element.setValue(names.get(i).toString(), values.get(i).toString());
         }
      }

      /**
       * Creates an object using data from an ObjectElement.
       *
       * @param element the element
       * @return the newly created object
       */
      public Object createObject(XMLControl element) {
         return new ControlTableModel();
      }

      /**
       * Loads an object with data from an ObjectElement.
       *
       * @param element the element
       * @param obj the object
       * @return the loaded object
       */
      public Object loadObject(XMLControl element, Object obj) {
         ControlTableModel model = (ControlTableModel) obj;
         // iterate over properties and add them to pts
         Iterator it = element.getPropertyNames().iterator();
         model.setLockValues(true);
         while(it.hasNext()) {
            String variable = (String) it.next();
            if(element.getPropertyType(variable).equals("string")) {
               model.setValue(variable, element.getString(variable));
            }
         }
         model.setLockValues(false);
         return obj;
      }
   }
}
