package org.opensourcephysics.controls;
import java.text.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class ControlTable extends JTable implements Control {

   ControlTableModel tableModel;
   CellTextField cellTextField = new CellTextField();
   Font font = null;
   public ControlTable(ControlTableModel tableModel) {
      super(tableModel);
      this.tableModel = tableModel;
      setRowHeight(22);
      //setRowMargin(2);
      setColumnModel(new ControlTableColumnModel());
      setDefaultRenderer(Object.class, new OSPCellRenderer());
      setDefaultEditor(Object.class, new OSPCellEditor(cellTextField));
   }

   /**
    * Enables the table so that it is editable.
    * @param enabled boolean
    */
   public void setEnabled(boolean enabled) {
      tableModel.setEnabled(enabled);
      super.setEnabled(enabled); // will firePropertyChange event
   }

   /**
    *  Adds a parameter to the input display.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    */
   public void setValue(String par, Object val) {
      tableModel.setValue(par, val);
      if(getEditingRow()==tableModel.names.indexOf(par)) {
         if(par instanceof String) {
            cellTextField.updateText((String) val);
         } else {
            cellTextField.updateText(tableModel.getString(par));
         }
      }
   }

   /**
    *  Adds an initial value of a parameter to the input display.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    */
   public void setValue(String par, double val) {
      tableModel.setValue(par, val);
      if(getEditingRow()==tableModel.names.indexOf(par)) {
         cellTextField.updateText(Double.toString(val));
      }
   }

   /**
    *  Adds an initial boolean value of a parameter to the input display.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    */
   public void setValue(String par, boolean val) {
      tableModel.setValue(par, val);
   }

   /**
    *  Adds an initial value of a parameter to the input display.
    *
    * @param  par  the parameter name
    * @param  val  the initial parameter value
    */
   public void setValue(String par, int val) {
      tableModel.setValue(par, val);
      if(getEditingRow()==tableModel.names.indexOf(par)) {
         cellTextField.updateText(Integer.toString(val));
      }
   }

   /**
    *  Removes a parameter from the table.
    *
    * @param  par  the parameter name
    */
   public void removeParameter(String par) {
      tableModel.removeParameter(par);
   }

   /**
    * Locks the control's interface. Values sent to the control will not
    * update the display until the control is unlocked.
    *
    * @param lock boolean
    */
   public void setLockValues(boolean lock) {
      tableModel.setLockValues(lock);
   }

   /**
    *  Creates a string representation of the control parameters.
    *
    * @return    the control parameters
    */
   public String toString() {
      return tableModel.toString();
   }

   /**
    * Gets the names of all properties stored in this control.
    *
    * @return List
    */
   public java.util.Collection getPropertyNames() {
      return tableModel.getPropertyNames();
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par
    * @return      double the value of of the parameter
    */
   public double getDouble(String par) {
      return tableModel.getDouble(par);
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par
    * @return      int the value of of the parameter
    */
   public int getInt(String par) {
      return tableModel.getInt(par);
   }

   /**
    * Gets the object with the specified property name.
    * Throws an UnsupportedOperationException if the named object has not been stored.
    *
    * @param name par
    * @return the object
    */
   public Object getObject(String par) throws UnsupportedOperationException {
      return tableModel.getObject(par);
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par  the parameter name
    * @return      String the value of of the parameter
    */
   public String getString(String par) {
      return tableModel.getString(par);
   }

   /**
    *  Reads a parameter value from the input display.
    *
    * @param  par  the parameter name
    * @return      the value of of the parameter
    */
   public boolean getBoolean(String par) {
      return tableModel.getBoolean(par);
   }

   /**
    *  Remove all text from the data input area.
    */
   public void clearValues() {
      tableModel.clearValues();
   }

   /**
    * println
    *
    * @param s String
    */
   public void println(String s) {
      tableModel.println(s);
   }

   /**
    * println
    */
   public void println() {}

   /**
    * print
    *
    * @param s String
    */
   public void print(String s) {
      tableModel.print(s);
   }

   /**
    * clearMessages
    */
   public void clearMessages() {}

   /**
    * calculationDone
    *
    * @param message String
    */
   public void calculationDone(String message) {
      tableModel.calculationDone(message);
   }

   class OSPCellEditor extends DefaultCellEditor {

      JTextField textField;
      OSPCellEditor(final JTextField textField) {
         super(textField);
         this.textField = textField;
         addCellEditorListener(new CellEditorListener() {

            public void editingCanceled(ChangeEvent e) {}

            public void editingStopped(ChangeEvent e) {
               //textField.setBackground(Color.WHITE);
            }
         });
      }
   }

   class CellTextField extends JTextField {

      int focusRow = -1, focusCol = -1;
      CellTextField() {
         enableEvents(AWTEvent.KEY_EVENT_MASK);
         enableEvents(AWTEvent.FOCUS_EVENT_MASK);
         enableEvents(AWTEvent.MOUSE_EVENT_MASK);
      }

      void updateText(String txt) {
         setText(txt);
         setBackground(Color.WHITE);
         tableModel.setColorAt(focusRow, Color.WHITE);
      }

      public void processMouseEvent( MouseEvent evt){
         if( evt.getClickCount()>1 && hasFocus() && focusRow>-1 && focusRow<tableModel.inspectorList.size()){
            OSPInspector inspector= (OSPInspector) tableModel.inspectorList.get(focusRow);
            if(inspector!=null){
               Object obj=inspector.show();  // inspector is modal
               // inspector may have created a new object
               tableModel.setValueAt(obj, focusRow, focusCol);
               // inspector may have changed object's data
               updateText(obj.toString());
            }
         }
         super.processMouseEvent(evt);
      }

      public void processKeyEvent(KeyEvent e) {
         //System.out.println("text key="+e.getKeyChar()+ " code="+ e.getKeyCode());
         // if (e.getKeyCode() == 10) { // enter key
         if(e.getKeyCode()=='\n') { // enter key
            setBackground(Color.WHITE);
            tableModel.setColorAt(focusRow, Color.WHITE);
            tableModel.setError(focusRow, false);
            //tableModel.fireTableCellUpdated(focusRow, 1);
         } else {
            setBackground(Color.YELLOW);
            tableModel.setColorAt(focusRow, Color.YELLOW);
         }
         super.processKeyEvent(e);
      }

      public void processFocusEvent(FocusEvent e) {
         if(FocusEvent.FOCUS_LOST==e.getID()) {
            if(focusRow<0 || focusRow>=tableModel.inspectorList.size() || tableModel.inspectorList.get(focusRow)==null){
              // update the text if isn't any inspector
              tableModel.setColorAt(focusRow, Color.WHITE);
              tableModel.setValueAt(this.getText(), focusRow, focusCol);
            }
            tableModel.fireTableCellUpdated(focusRow, 1);
         }
         focusRow = getEditingRow();
         focusCol = getEditingColumn();
         focusCol = convertColumnIndexToModel(focusCol);
         if((focusRow>0)&&(focusCol>0)) {
            setBackground(tableModel.getColorAt(focusRow, focusCol));
         }
         super.processFocusEvent(e);
      }
   }

   class OSPCellRenderer extends DefaultTableCellRenderer {

      NumberFormat numberFormat = NumberFormat.getInstance();
      public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
          int row, int column) {
         column = table.convertColumnIndexToModel(column);
         boolean rowEditable = tableModel.isCellEditable(row, 1);
         if((column==0)||!rowEditable) {
            hasFocus = false;
         }
         super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
         OSPInspector inspector= (OSPInspector)tableModel.inspectorList.get(row);
         if((column==1)&&tableModel.hasError(row)) {
            setForeground(Color.BLACK);
            setBackground(ControlTableModel.ERROR_RED);
         } else if((column==0)||!rowEditable) {
            setForeground(Color.BLACK);
            setBackground(ControlTableModel.PANEL_BACKGROUND);
         } else if(column==1) {
            setForeground(isSelected
                          ? Color.BLUE
                          : Color.BLACK);
            setBackground(tableModel.getColorAt(row, column));
         }
         if (inspector!=null&&column>0){
            setBackground(inspector.highlightColor);
         }
         if(font!=null) {
            setFont(font);
         }
         numberFormat.setMaximumFractionDigits(8);
         return this;
      }

      public void setDecimalFormat(String pattern) {
         numberFormat = new DecimalFormat(pattern);
      }

      public void setMaximumFractionDigits(int digits) {
         numberFormat.setMaximumFractionDigits(digits);
      }

      public void setValue(Object value) {
         if(value==null) {
            setText("");
         } else if(value instanceof Double) {
            setText(numberFormat.format(value));
         } else {
            setText(value.toString());
         }
      }
   }

   class ControlTableColumnModel extends DefaultTableColumnModel {

      /**
       * Fixes bug in DefaultTableColumnModel.
       * @param columnIndex int
       * @return TableColumn
       */
      public TableColumn getColumn(int columnIndex) {
         if(tableColumns==null || columnIndex>=tableColumns.size()) {
            return null;
         } else {
            return super.getColumn(columnIndex);
         }
      }
   }

   /**
    * Returns an XML.ObjectLoader to save and load data for this object.
    *
    * @return the object loader
    */
   public static XML.ObjectLoader getLoader() {
      return new ControlTableLoader();
   }

   /**
    * A class to save and load data for OSPControls.
    */
   static class ControlTableLoader implements XML.ObjectLoader {

      /**
       * Saves object data to an XMLControl.
       *
       * @param control the control to save to
       * @param obj the object to save
       */
      public void saveObject(XMLControl control, Object obj) {
         ControlTable table = (ControlTable) obj;
         ArrayList names = (ArrayList) table.tableModel.names.clone();
         ArrayList values = (ArrayList) table.tableModel.values.clone();
         for(int i = 0, n = names.size(); i<n; i++) {
            control.setValue(names.get(i).toString(), values.get(i).toString());
         }
      }

      /**
       * Creates an object using data from an XMLControl.
       *
       * @param control the control
       * @return the newly created object
       */
      public Object createObject(XMLControl control) {
         return new ControlTable(new ControlTableModel());
      }

      /**
       * Loads an object with data from an XMLControl.
       *
       * @param control the control
       * @param obj the object
       * @return the loaded object
       */
      public Object loadObject(XMLControl control, Object obj) {
         ControlTable table = (ControlTable) obj;
         // iterate over properties and add them to table model
         Iterator it = control.getPropertyNames().iterator();
         table.tableModel.setLockValues(true);
         while(it.hasNext()) {
            String variable = (String) it.next();
            if(control.getPropertyType(variable).equals("string")) {
               table.tableModel.setValue(variable, control.getString(variable));
            }
         }
         table.tableModel.setLockValues(false);
         return obj;
      }
   }
}
