/*
 *  The org.opensourcephysics.controls package defines the framework for building
 *  user interface controls for the book Simulations in Physics.
 *  Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 *
 */
package org.opensourcephysics.controls;

import java.awt.event.*;
import javax.swing.*;
import org.opensourcephysics.display.*;

/**
 *  A Control class for Calculations. The GUI consisting of an input text area
 *  and calculate and reset buttons designed to control a Calculation.
 *
 * @author       Wolfgang Christian
 * @author       Joshua Gould
 * @version 1.0
 */
public class CalculationControl extends OSPControl {
  protected Calculation calculation;
  final static String resetToolTipText = ControlsRes.CALCULATION_RESET_TIP;
  final static String calcToolTipText = ControlsRes.CALCULATION_CALC_TIP;
  JButton calcBtn;
  JButton resetBtn;

  /**
   *  The CalculationControl constructor.
   *
   * @param  _calculation
   */
  public CalculationControl(Calculation _calculation) {
    super(_calculation);
    calculation = _calculation;
    if (model!=null){
       String name = model.getClass().getName();
       setTitle(name.substring(1+name.lastIndexOf("."))+" Control");
    }
    calcBtn = new JButton(ControlsRes.CALCULATION_CALC);
    calcBtn.setToolTipText(calcToolTipText);
    resetBtn = new JButton(ControlsRes.CALCULATION_RESET);
    resetBtn.setToolTipText(resetToolTipText);
    calcBtn.addActionListener(new CalcBtnListener());
    resetBtn.addActionListener(new ResetBtnListener());
    buttonPanel.add(calcBtn);
    buttonPanel.add(resetBtn);
    validate();
    pack();
  }

  private class CalcBtnListener implements ActionListener {

    /**
     *  Performs the calculation and shows all drawing frames.
     *
     * @param  e
     */
    public void actionPerformed(ActionEvent e) {
      GUIUtils.clearDrawingFrameData(false);
      if(calculation==null) {
        println("This CalculationControl's model is null.");
        return;
      }
      calculation.calculate();
      org.opensourcephysics.display.GUIUtils.showDrawingAndTableFrames();
    }
  }
  private class ResetBtnListener implements ActionListener {

    /**
     *  Resets the calculation.
     *
     * @param  e
     */
    public void actionPerformed(ActionEvent e) {
      GUIUtils.clearDrawingFrameData(true);
      if(calculation==null) {
        println("This CalculationControl's model is null.");
        return;
      }
      calculation.resetCalculation();
      if(xmlDefault!=null) {
        xmlDefault.loadObject(getOSPApp());
      }
    }
  }

  /**
   * Creates a calculation control and establishes communication between the control and the model.
   *
   * @param animation Animation
   * @return AnimationControl
   */
  public static CalculationControl createApp(Calculation model) {
    CalculationControl control = new CalculationControl(model);
    model.setControl(control);
    return control;
  }

  /**
   * Creates a calculation control and establishes communication between the control and the model.
   * Initial parameters are set using the xml data.
   *
   * @param model Calculation
   * @param xml String[]
   * @return CalculationControl
   */
  public static CalculationControl createApp(Calculation model, String[] xml) {
    CalculationControl control = createApp(model);
    control.loadXML(xml);
    return control;
  }
}
