/*
 *  The org.opensourcephysics.controls package defines the framework for building
 *  user interface controls for the book Simulations in Physics.
 *  Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.JButton;

/**
 *  A GUI consisting of an input text area, a message area, and various buttons
 *  to initialize and control an Animation.
 *
 * @author       Wolfgang Christian
 * @author       Joshua Gould
 * @version 1.0
 */

public class AnimationTextControl extends TextControl {
  protected Animation animation;
  final static String resetToolTipText = ControlsRes.ANIMATION_RESET_TIP;
  final static String initToolTipText = ControlsRes.ANIMATION_INIT_TIP;
  final static String startToolTipText = ControlsRes.ANIMATION_START_TIP;
  final static String stopToolTipText = ControlsRes.ANIMATION_STOP_TIP;
  final static String newToolTipText = ControlsRes.ANIMATION_NEW_TIP;
  final static String stepToolTipText = ControlsRes.ANIMATION_STEP_TIP;
  boolean stepModeEditing = true; // enables input editing while single stepping
  ArrayList customButtons = new ArrayList(); // list of custom buttons, custom buttons are enabled when the animation is stopped
  JButton startBtn = new JButton(ControlsRes.ANIMATION_INIT); // changes to start, stop
  JButton stepBtn = new JButton(ControlsRes.ANIMATION_STEP);
  JButton resetBtn = new JButton(ControlsRes.ANIMATION_RESET); // changes to new

  /**
   *  AnimationTextControl constructor.
   *
   * @param  _animation  the animation for this animation.
   */
  public AnimationTextControl(Animation _animation) {
    super(_animation);
    animation = _animation;
    startBtn.addActionListener(new StartBtnListener());
    startBtn.setToolTipText(initToolTipText);
    stepBtn.addActionListener(new StepBtnListener());
    stepBtn.setToolTipText(stepToolTipText);
    resetBtn.addActionListener(new ResetBtnListener());
    resetBtn.setToolTipText(resetToolTipText);
    stepBtn.setEnabled(false);
    buttonPanel.add(startBtn);
    buttonPanel.add(stepBtn);
    buttonPanel.add(resetBtn);
    validate();
    pack();
  }

  /**
   *  Adds a custom button to the control's frame.
   *
   * @param  methodName   the name of the method in the Animation to be invoked.
   *      The method must have no parameters.
   * @param  text         the button's text label
   * @param  toolTipText  the button's tool tip text
   * @return              the custom button
   */
  public JButton addButton(String methodName, String text, String toolTipText) {
    JButton b = super.addButton(methodName, text, toolTipText);
    customButtons.add(b);
    return b;
  }

  /**
   *  Signals the control that the animation has completed.
   *  The control should reset itself in preparation for a new
   *  animation. The given message is printed in the message area.
   *
   * @param  message
   */
  public void calculationDone(String message) {
    startBtnActionPerformed(new ActionEvent(this, 0, ControlsRes.ANIMATION_STOP));
    resetBtnActionPerformed(new ActionEvent(this, 0, ControlsRes.ANIMATION_NEW));
    resetBtn.setEnabled(true);
    println(message);
  }

  /**
   * Method startBtnActionPerformed
   *
   * @param e
   */
  void startBtnActionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals(ControlsRes.ANIMATION_INIT)) {
      stepBtn.setEnabled(true);
      startBtn.setText(ControlsRes.ANIMATION_START);
      startBtn.setToolTipText(startToolTipText);
      resetBtn.setText(ControlsRes.ANIMATION_NEW);
      resetBtn.setToolTipText(newToolTipText);
      resetBtn.setEnabled(true);
      inputTextArea.setEditable(stepModeEditing);
      readItem.setEnabled(stepModeEditing);
      inputTextArea.setEnabled(stepModeEditing);
      messageTextArea.setEditable(false);
      org.opensourcephysics.display.GUIUtils.showDrawingAndTableFrames();
      animation.initializeAnimation();
    }
    else if (e.getActionCommand().equals(ControlsRes.ANIMATION_START)) {
      setCustomButtonsEnabled(false);
      startBtn.setText(ControlsRes.ANIMATION_STOP);
      startBtn.setToolTipText(stopToolTipText);
      stepBtn.setEnabled(false);
      resetBtn.setEnabled(false);
      readItem.setEnabled(false);
      inputTextArea.setEditable(false);
      inputTextArea.setEnabled(false);
      animation.startAnimation();
    }
    else { // action command = Stop
      startBtn.setText(ControlsRes.ANIMATION_START);
      setCustomButtonsEnabled(true);
      startBtn.setToolTipText(startToolTipText);
      stepBtn.setEnabled(true);
      resetBtn.setEnabled(true);
      readItem.setEnabled(stepModeEditing);
      inputTextArea.setEditable(stepModeEditing);
      inputTextArea.setEnabled(stepModeEditing);
      animation.stopAnimation();
    }
  }

  /**
   * Method resetBtnActionPerformed
   *
   * @param e
   */
  void resetBtnActionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals(ControlsRes.ANIMATION_RESET)) {
      animation.resetAnimation();
    }
    else { //action command = New
      startBtn.setText(ControlsRes.ANIMATION_INIT);
      startBtn.setToolTipText(initToolTipText);
      resetBtn.setText(ControlsRes.ANIMATION_RESET);
      resetBtn.setToolTipText(resetToolTipText);
      stepBtn.setEnabled(false);
      readItem.setEnabled(true);
      inputTextArea.setEditable(true);
      inputTextArea.setEnabled(true);
      messageTextArea.setEditable(true);
      setCustomButtonsEnabled(true);
    }
  }

  /**
   * Method stepBtnActionPerformed
   *
   * @param e
   */
  void stepBtnActionPerformed(ActionEvent e) {
    animation.stepAnimation();
  }

  private void setCustomButtonsEnabled(boolean enabled) {
    if (customButtons != null) {
      for (Iterator it = customButtons.iterator(); it.hasNext(); ) {
        ( (JButton) it.next()).setEnabled(enabled);
      }
    }
  }

  /**
   * Class StartBtnListener
   */
  class StartBtnListener
      implements ActionListener {

    /**
     * Method actionPerformed
     *
     * @param e
     */
    public void actionPerformed(ActionEvent e) {
      startBtnActionPerformed(e);
    }
  }

  /**
   * Class ResetBtnListener
   */
  class ResetBtnListener
      implements ActionListener {

    /**
     * Method actionPerformed
     *
     * @param e
     */
    public void actionPerformed(ActionEvent e) {
      resetBtnActionPerformed(e);
    }
  }

  /**
   * Class StepBtnListener
   */
  class StepBtnListener
      implements ActionListener {

    /**
     * Method actionPerformed
     *
     * @param e
     */
    public void actionPerformed(ActionEvent e) {
      stepBtnActionPerformed(e);
    }
  }

  /**
   * Returns an XML.ObjectLoader to save and load data for this object.
   *
   * @return the object loader
   */
  public static XML.ObjectLoader getLoader() {
    return new AnimationTextControlLoader();
  }


  /**
   * A class to save and load data for OSPControls.
   */
  static class AnimationTextControlLoader extends TextControlLoader {
    /**
     * Saves object data to an ObjectElement.
     *
     * @param element the element to save to
     * @param obj the object to save
     */
    public void saveObject(XMLControl element, Object obj) {
      AnimationTextControl control = (AnimationTextControl) obj;
      if (control.startBtn.getText().equals(ControlsRes.ANIMATION_STOP)) {
        control.startBtn.doClick(); // stop the animation if it is running
      }
      element.setValue("initialize_mode", control.startBtn.getText().equals(ControlsRes.ANIMATION_INIT));
      super.saveObject(element, obj);
    }

    /**
     * Creates an object using data from an ObjectElement.
     *
     * @param element the element
     * @return the newly created object
     */
    public Object createObject(XMLControl element) {
      return new AnimationTextControl(null);
    }

    /**
     * Loads an object with data from an ObjectElement.
     *
     * @param element the element
     * @param obj the object
     * @return the loaded object
     */
    public Object loadObject(XMLControl element, Object obj) {
      AnimationTextControl animationControl = (AnimationTextControl) obj;
      if (animationControl.startBtn.getText().equals(ControlsRes.ANIMATION_STOP)) {
        animationControl.startBtn.doClick(); // stop the animation if it is running
      }
      if (!animationControl.startBtn.getText().equals(ControlsRes.ANIMATION_INIT)) {
        animationControl.resetBtn.doClick(); // put the system into the initialize state
      }
      if(!element.getBoolean("initialize_mode")){
       animationControl.startBtn.doClick();
      }
      // the control's buttons are now correct
      super.loadObject(element, obj); // load the control's parameters and the model
      animationControl.clearMessages();
      return obj;
    }
  }

}
