
package org.opensourcephysics.controls;
import java.awt.*;

/**
 * AbstractSimulation is a template for SIP animations.
 *
 * AbstractSimulation creates and manages an animation thread that
 * invokes the abstract "doStep()" method every 1/10 second.  The doStep method is also called when
 * the stepAnimation button is pressed.
 *
 * Implement the doStep method to create a concrete simulation.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
abstract public class AbstractSimulation extends AbstractAnimation implements Simulation {

   protected SimulationControl control; // shadow the previous definition
   protected boolean showStepsPerDisplay = false;
   protected int stepsPerDisplay = 1;
   protected int stepCounter = 0;

   /**
    * Sets the Control for this model and initializes the control's values.
    *
    * @param control
    */
   public void setControl(Control control) {
      this.control = (SimulationControl) control;
      super.control = control;
      if(control!=null) {
         resetAnimation();
         if(control instanceof Frame) {
            ((Frame) control).pack();
         }
      }
   }

   /**
    * Performs an action before executing one or more animation steps.
    */
   public void startRunning() {}

   /**
    * Performs an action after executing one or more animation steps.
    */
   public void stopRunning() {}

   /**
    * Invokes the simulation's start method and then starts the animation thread.
    * Simulations should invoke the start method.
    *
    * @deprecated
    */
   public void startAnimation() {
      if(showStepsPerDisplay) {
         stepsPerDisplay = control.getInt("steps per display");
      }
      start();
      super.startAnimation();
   }

   /**
    * Steps the animation.
    *
    */
   public void stepAnimation() {
      if(showStepsPerDisplay) {
         stepsPerDisplay = control.getInt("steps per display");
      }
      super.stepAnimation();
      stepCounter++;
      org.opensourcephysics.display.GUIUtils.repaintOSPFrames();
   }

   /**
    * Starts the simulation.
    *
    * Override this method to perform custom actions before the animation thread begins running.
    */
   public void start() {}

   /**
    * Stops the animation thread and then invokes the simulations stop method.\
    * Simulations should invoke the stop method.
    *
    * @deprecated
    */
   public void stopAnimation() {
      super.stopAnimation();
      stop();
   }

   /**
    * Stops the simulation.
    *
    * Override this method to perform custom actions after the animation thread stops running.
    */
   public void stop() {}

   /**
    * Initializes the animation.
    * Simulations should invoke the initialize method.
    *
    * @deprecated
    */
   public void initializeAnimation() {
      if(control==null) {
         return; // control can be null in applet mode so check for this
      }
      super.initializeAnimation();
      initialize();
      stepCounter = 0;
   }

   /**
    * Gets number of animation steps that have been performed since the last initializeAnimation.
    *
    * @return stepCounter
    */
   public int getStepCounter() {
      return stepCounter;
   }

   /**
    * Initializes the simulation.
    *
    * Override this method to initialization the simulation.
    */
   public void initialize() {}

   /**
    * Resets the animation to its default condition.
    * Simulations should invoke the reset method.
    *
    * @deprecated
    */
   public void resetAnimation() {
      if(control==null) {
         return; // control can be null in applet mode so check for this
      }
      super.resetAnimation();
      stepsPerDisplay = 1;
      if(showStepsPerDisplay) {
         control.setAdjustableValue("steps per display", stepsPerDisplay);
      }
      reset();
   }

   /**
    * Enables the steps per display variable in the control;
    * @param enable boolean
    */
   public void enableStepsPerDisplay(boolean enable) {
      showStepsPerDisplay = enable;
      if(showStepsPerDisplay) {
         control.setAdjustableValue("steps per display", stepsPerDisplay);
      } else {
         control.removeParameter("steps per display");
      }
   }

   /**
    * Sets the number of animation steps before animated drawing panels are rendered.
    *
    * The default steps per animation is 1.  Increase this number if frequent rendering
    * causes slugish behavior.
    *
    * @param num int
    */
   public void setStepsPerDisplay(int num) {
      stepsPerDisplay = Math.max(num, 1);
      if(showStepsPerDisplay) {
         control.setAdjustableValue("steps per display", stepsPerDisplay);
      }
   }

   /**
    * Resets the simulation to its default state.
    *
    * Override this method to set the simulation's parameters.
    */
   public void reset() {}

   /**
    * Implementation of Runnable interface.  DO NOT access this method directly.
    */
   public void run() {
      long sleepTime = delayTime;
      while(animationThread==Thread.currentThread()) {
         long currentTime = System.currentTimeMillis();
         for(int i = 0; i<stepsPerDisplay; i++) {
            doStep();
            stepCounter++;
            if(animationThread!=Thread.currentThread()) {
               break;          // check for stop condition
            } else {
               Thread.yield(); // give other threads a chance to run if needed
            }
         }
         org.opensourcephysics.display.GUIUtils.renderOSPFrames();
         //adjust the sleep time to try and achieve a constant animation rate
         sleepTime = Math.max(1, delayTime-(System.currentTimeMillis()-currentTime));
         try {
            Thread.sleep(sleepTime);
         } catch(InterruptedException ie) {}
      }
   }
}
